/* SPDX-License-Identifier: MIT */
/**
	@file		ntv2formatdescriptor.cpp
	@brief		Implementation of the NTV2FormatDescriptor class.
	@copyright	(C) 2016-2022 AJA Video Systems, Inc.
**/
#include "ntv2formatdescriptor.h"
#include "ntv2utils.h"
#if defined(AJALinux)
	#include <string.h>  // For memset
	#include <stdint.h>
#endif
#include <sstream>
#include <iomanip>


using namespace std;

#define	AJA_FD_BEGIN			NTV2FormatDescriptor (
#define	AJA_FD_END				0)
#define	AJA_FD_NOTSUPPORTED		NTV2FormatDescriptor ()
#define	AJA_FD_TBD				NTV2FormatDescriptor ()

#define	AsConstUBytePtr(_x_)	reinterpret_cast<const UByte*>(_x_)
#define	AsUBytePtr(_x_)			reinterpret_cast<UByte*>(_x_)


static const string	gPlaneLabels[NTV2_FBF_NUMFRAMEBUFFERFORMATS][4]	=
{
	/* NTV2_FBF_10BIT_YCBCR */				{"YCbCr",	"",		"",		""},
	/* NTV2_FBF_8BIT_YCBCR */				{"YCbCr",	"",		"",		""},
	/* NTV2_FBF_ARGB */						{"ARGB",	"",		"",		""},
	/* NTV2_FBF_RGBA */						{"RGBA",	"",		"",		""},
	/* NTV2_FBF_10BIT_RGB */				{"RGB",		"",		"",		""},
	/* NTV2_FBF_8BIT_YCBCR_YUY2 */			{"YCbCr",	"",		"",		""},
	/* NTV2_FBF_ABGR */						{"ABGR",	"",		"",		""},
	/* NTV2_FBF_10BIT_DPX */				{"",		"",		"",		""},
	/* NTV2_FBF_10BIT_YCBCR_DPX */			{"YCbCr",	"",		"",		""},
	/* NTV2_FBF_8BIT_DVCPRO */				{"",		"",		"",		""},
	/* NTV2_FBF_8BIT_YCBCR_420PL3 */		{"Y",		"Cb",	"Cr",	""},
	/* NTV2_FBF_8BIT_HDV */					{"",		"",		"",		""},
	/* NTV2_FBF_24BIT_RGB */				{"RGB",		"",		"",		""},
	/* NTV2_FBF_24BIT_BGR */				{"BGR",		"",		"",		""},
	/* NTV2_FBF_10BIT_YCBCRA */				{"YCbCrA",	"",		"",		""},
	/* NTV2_FBF_10BIT_DPX_LE */				{"",		"",		"",		""},
	/* NTV2_FBF_48BIT_RGB */				{"RGB",		"",		"",		""},
	/* NTV2_FBF_12BIT_RGB_PACKED */			{"RGB",		"",		"",		""},
	/* NTV2_FBF_PRORES_DVCPRO */			{"",		"",		"",		""},
	/* NTV2_FBF_PRORES_HDV */				{"",		"",		"",		""},
	/* NTV2_FBF_10BIT_RGB_PACKED */			{"RGB",		"",		"",		""},
	/* NTV2_FBF_10BIT_ARGB */				{"ARGB",	"",		"",		""},
	/* NTV2_FBF_16BIT_ARGB */				{"ARGB",	"",		"",		""},
	/* NTV2_FBF_8BIT_YCBCR_422PL3 */		{"Y",		"Cb",	"Cr",	""},
	/* NTV2_FBF_10BIT_RAW_RGB */			{"RGB",		"",		"",		""},
	/* NTV2_FBF_10BIT_RAW_YCBCR */			{"YCbCr",	"",		"",		""},
	/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	{"Y",		"Cb",	"Cr",	""},
	/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	{"Y",		"Cb",	"Cr",	""},
	/* NTV2_FBF_10BIT_YCBCR_420PL2 */		{"Y",		"CbCr",	"",		""},
	/* NTV2_FBF_10BIT_YCBCR_422PL2 */		{"Y",		"CbCr",	"",		""},
	/* NTV2_FBF_8BIT_YCBCR_420PL2 */		{"Y",		"CbCr",	"",		""},
	/* NTV2_FBF_8BIT_YCBCR_422PL2 */		{"Y",		"CbCr",	"",		""}
};


static const uint8_t	gBitsPerComponent[NTV2_FBF_NUMFRAMEBUFFERFORMATS][3]	=
{	//										Luma		Chroma	Alpha
	/* NTV2_FBF_10BIT_YCBCR */				{10,		10,		0},
	/* NTV2_FBF_8BIT_YCBCR */				{8,			8,		0},
	/* NTV2_FBF_ARGB */						{0,			8,		8},
	/* NTV2_FBF_RGBA */						{0,			8,		8},
	/* NTV2_FBF_10BIT_RGB */				{0,			10,		2},
	/* NTV2_FBF_8BIT_YCBCR_YUY2 */			{8,			8,		0},
	/* NTV2_FBF_ABGR */						{0,			8,		8},
	/* NTV2_FBF_10BIT_DPX */				{0,			10,		0},
	/* NTV2_FBF_10BIT_YCBCR_DPX */			{10,		10,		0},
	/* NTV2_FBF_8BIT_DVCPRO */				{0,			0,		0},	//	Lossy
	/* NTV2_FBF_8BIT_YCBCR_420PL3 */		{8,			8,		0},
	/* NTV2_FBF_8BIT_HDV */					{0,			0,		0},	//	Lossy
	/* NTV2_FBF_24BIT_RGB */				{0,			8,		0},
	/* NTV2_FBF_24BIT_BGR */				{0,			8,		0},
	/* NTV2_FBF_10BIT_YCBCRA */				{0,			0,		0},	//	Unused
	/* NTV2_FBF_10BIT_DPX_LE */				{0,			10,		0},
	/* NTV2_FBF_48BIT_RGB */				{0,			16,		0},
	/* NTV2_FBF_12BIT_RGB_PACKED */			{0,			12,		0},
	/* NTV2_FBF_PRORES_DVCPRO */			{0,			0,		0},	//	Unused
	/* NTV2_FBF_PRORES_HDV */				{0,			0,		0},	//	Unused
	/* NTV2_FBF_10BIT_RGB_PACKED */			{0,			0,		0},	//	Unused
	/* NTV2_FBF_10BIT_ARGB */				{0,			0,		0},	//	Unused
	/* NTV2_FBF_16BIT_ARGB */				{0,			0,		0},	//	Unused
	/* NTV2_FBF_8BIT_YCBCR_422PL3 */		{8,			8,		0},
	/* NTV2_FBF_10BIT_RAW_RGB */			{0,			10,		0},
	/* NTV2_FBF_10BIT_RAW_YCBCR */			{10,		10,		0},
	/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	{10,		10,		0},
	/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	{10,		10,		0},
	/* NTV2_FBF_10BIT_YCBCR_420PL2 */		{10,		10,		0},
	/* NTV2_FBF_10BIT_YCBCR_422PL2 */		{10,		10,		0},
	/* NTV2_FBF_8BIT_YCBCR_420PL2 */		{8,			8,		0},
	/* NTV2_FBF_8BIT_YCBCR_422PL2 */		{8,			8,		0}
};


const static NTV2FormatDescriptor	formatDescriptorTable [NTV2_NUM_STANDARDS] [NTV2_FBF_NUMFRAMEBUFFERFORMATS] =
{
	{/////	NTV2_STANDARD_1080											inNumLines						inNumPixels							inLinePitch [ULWords per line]			inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_YCBCRLINEPITCH_1080,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080*2/4,			AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080*2/4,			AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_YCBCRLINEPITCH_1080,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080_DVCPRO,	HD_YCBCRLINEPITCH_1080_DVCPRO,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080_HDV,		HD_YCBCRLINEPITCH_1080_HDV,				AJA_FD_END,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			RGB24LINEPITCH_1080,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			RGB24LINEPITCH_1080,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			RGB48LINEPITCH_1080,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			RGB12PLINEPITCH_1080,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080*4/3,			AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080*2,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080/4,			AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080*5/16,		AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080*5/16,		AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080/4,			AJA_FD_END
	},
	{/////	NTV2_STANDARD_720											inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_YCBCRLINEPITCH_720,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720*2/4,			AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720,				AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720*2/4,			AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_YCBCRLINEPITCH_720,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720_DVCPRO,	HD_YCBCRLINEPITCH_720_DVCPRO,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720_HDV,		HD_YCBCRLINEPITCH_720_HDV,				AJA_FD_END,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			RGB24LINEPITCH_720,						AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			RGB24LINEPITCH_720,						AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720,				AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			RGB48LINEPITCH_720,						AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			RGB12PLINEPITCH_720,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720,				AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720*4/3,			AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720*2,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720/4,			AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_720,			HD_NUMCOMPONENTPIXELS_720,			HD_NUMCOMPONENTPIXELS_720/4,			AJA_FD_END
	},
	{/////	NTV2_STANDARD_525											inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					YCBCRLINEPITCH_SD,						AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS*2/4,					AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS*2/4,					AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					YCBCRLINEPITCH_SD,						AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			480,							NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS/4,					AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					RGB24LINEPITCH_525,						AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					RGB24LINEPITCH_525,						AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					RGB48LINEPITCH_525,						AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					RGB12PLINEPITCH_525,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS*4/3,					AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			NUMACTIVELINES_525,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS*2,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			480,							NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS/4,					AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			480,							NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS/2,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			480,							NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS/2,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			480,							NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS*5/16,				AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			480,							NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS*5/16,				AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			480,							NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS/4,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			480,							NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS/4,					AJA_FD_END
	},
	{/////	NTV2_STANDARD_625											inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					YCBCRLINEPITCH_SD,						AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS*2/4,					AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS*2/4,					AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					YCBCRLINEPITCH_SD,						AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS/4,					AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					RGB24LINEPITCH_625,						AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					RGB24LINEPITCH_625,						AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					RGB48LINEPITCH_625,						AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					RGB12PLINEPITCH_625,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS,						AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS*4/3,					AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS*2,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS/4,					AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS/2,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS/2,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS*5/16,				AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS*5/16,				AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS/4,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			NUMACTIVELINES_625,				NUMCOMPONENTPIXELS,					NUMCOMPONENTPIXELS/4,					AJA_FD_END
	},
	{/////	NTV2_STANDARD_1080p											inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_YCBCRLINEPITCH_1080,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080*2/4,			AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080*2/4,			AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_YCBCRLINEPITCH_1080,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080_DVCPRO,	HD_YCBCRLINEPITCH_1080_DVCPRO,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080_HDV,		HD_YCBCRLINEPITCH_1080_HDV,				AJA_FD_END,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			RGB24LINEPITCH_1080,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			RGB24LINEPITCH_1080,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			RGB48LINEPITCH_1080,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			RGB12PLINEPITCH_1080,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080,				AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080*4/3,			AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080*2,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080/4,			AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			((HD_NUMCOMPONENTPIXELS_1080*10)/8)/4,	AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			((HD_NUMCOMPONENTPIXELS_1080*10)/8)/4,	AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080*5/16,		AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080*5/16,		AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_1080,			HD_NUMCOMPONENTPIXELS_1080/4,			AJA_FD_END
	},
	{/////	NTV2_STANDARD_2K (OBSOLETE)									inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_YCBCRLINEPITCH_2K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_YCBCRLINEPITCH_2K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			RGB24LINEPITCH_2048,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			RGB24LINEPITCH_2048,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			RGB48LINEPITCH_2048,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			RGB12PLINEPITCH_2048,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*4/3,			AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*2,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/4,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			((HD_NUMCOMPONENTPIXELS_2K*10)/8)/4,	AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			((HD_NUMCOMPONENTPIXELS_2K*10)/8)/4,	AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_2K,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/4,				AJA_FD_END
	},
	{/////	NTV2_STANDARD_2Kx1080p										inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_YCBCRLINEPITCH_2K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_YCBCRLINEPITCH_2K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			RGB24LINEPITCH_2048,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			RGB24LINEPITCH_2048,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			RGB48LINEPITCH_2048,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			RGB12PLINEPITCH_2048,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*4/3,			AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*2,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/4,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/4,				AJA_FD_END
	},
	{/////	NTV2_STANDARD_2Kx1080i										inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_YCBCRLINEPITCH_2K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_YCBCRLINEPITCH_2K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			RGB24LINEPITCH_2048,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			RGB24LINEPITCH_2048,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			RGB48LINEPITCH_2048,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			RGB12PLINEPITCH_2048,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*4/3,			AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*2,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/4,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMACTIVELINES_1080,			HD_NUMCOMPONENTPIXELS_2K,			HD_NUMCOMPONENTPIXELS_2K/4,				AJA_FD_END
	},
	{/////	NTV2_STANDARD_3840x2160p									inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_YCBCRLINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*2/4,		AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*2/4,		AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_YCBCRLINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		RGB24LINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		RGB24LINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		RGB48LINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		RGB12PLINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*4/3,		AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*2,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/4,			AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*5/16,		AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*5/16,		AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/4,			AJA_FD_END
	},
	{/////	NTV2_STANDARD_4096x2160p									inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_YCBCRLINEPITCH_4K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_YCBCRLINEPITCH_4K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			RGB24LINEPITCH_4096,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			RGB24LINEPITCH_4096,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			RGB48LINEPITCH_4096,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			RGB12PLINEPITCH_4096,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*4/3,			AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*2,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/4,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/4,				AJA_FD_END
	},
	{/////	NTV2_STANDARD_3840HFR										inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_YCBCRLINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*2/4,		AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*2/4,		AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_YCBCRLINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		RGB24LINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		RGB24LINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		RGB48LINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		RGB12PLINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*4/3,		AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*2,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/4,			AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*5/16,		AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*5/16,		AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/4,			AJA_FD_END
	},
	{/////	NTV2_STANDARD_4096HFR										inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_YCBCRLINEPITCH_4K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_YCBCRLINEPITCH_4K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			RGB24LINEPITCH_4096,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			RGB24LINEPITCH_4096,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			RGB48LINEPITCH_4096,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			RGB12PLINEPITCH_4096,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*4/3,			AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*2,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/4,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/4,				AJA_FD_END
	},
	{/////	NTV2_STANDARD_7680											inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_YCBCRLINEPITCH_UHD2,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2*2/4,			AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2,				AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2*2/4,			AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_YCBCRLINEPITCH_UHD2,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			RGB24LINEPITCH_7680,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			RGB24LINEPITCH_7680,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2,				AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			RGB48LINEPITCH_7680,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			RGB12PLINEPITCH_7680,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2*4/3,			AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2*2,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2/4,			AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2*5/16,		AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2*5/16,		AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_UHD2,			FD_NUMCOMPONENTPIXELS_UHD2/4,			AJA_FD_END
	},
	{/////	NTV2_STANDARD_8192											inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_YCBCRLINEPITCH_8K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K,				AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_YCBCRLINEPITCH_8K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			RGB24LINEPITCH_8192,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			RGB24LINEPITCH_8192,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K,				AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			RGB48LINEPITCH_8192,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			RGB12PLINEPITCH_8192,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K*4/3,			AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K*2,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K/4,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			FD_NUMLINES_8K,					FD_NUMCOMPONENTPIXELS_8K,			FD_NUMCOMPONENTPIXELS_8K/4,				AJA_FD_END
	},
	{/////	NTV2_STANDARD_3840i											inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_YCBCRLINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*2/4,		AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*2/4,		AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_YCBCRLINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		RGB24LINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		RGB24LINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		RGB48LINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		RGB12PLINEPITCH_3840,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD,			AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*4/3,		AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*2,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/4,			AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/2,			AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*5/16,		AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD*5/16,		AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/4,			AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_QUADHD,		HD_NUMCOMPONENTPIXELS_QUADHD/4,			AJA_FD_END
	},
	{/////	NTV2_STANDARD_4096i											inNumLines						inNumPixels							inLinePitch								inFirstActiveLine
		/* NTV2_FBF_10BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_YCBCRLINEPITCH_4K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ARGB */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_RGBA */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_YUY2 */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*2/4,			AJA_FD_END,
		/* NTV2_FBF_ABGR */						AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_DPX */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_YCBCRLINEPITCH_4K,					AJA_FD_END,
		/* NTV2_FBF_8BIT_DVCPRO */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_8BIT_YCBCR_420PL3 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_HDV */					AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_24BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			RGB24LINEPITCH_4096,					AJA_FD_END,
		/* NTV2_FBF_24BIT_BGR */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			RGB24LINEPITCH_4096,					AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCRA */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_DPX_LE */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_48BIT_RGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			RGB48LINEPITCH_4096,					AJA_FD_END,
		/* NTV2_FBF_12BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			RGB12PLINEPITCH_4096,					AJA_FD_END,
		/* NTV2_FBF_PRORES_DVCPRO */			AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_PRORES_HDV */				AJA_FD_NOTSUPPORTED,
		/* NTV2_FBF_10BIT_RGB_PACKED */			AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K,				AJA_FD_END,
		/* NTV2_FBF_10BIT_ARGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*4/3,			AJA_FD_END,
		/* NTV2_FBF_16BIT_ARGB */				AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*2,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL3 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/4,				AJA_FD_END,
		/* NTV2_FBF_10BIT_RAW_RGB */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_RAW_YCBCR */			AJA_FD_TBD,
		/* NTV2_FBF_10BIT_YCBCR_420PL3_LE */	AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_422PL3_LE */	AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/2,				AJA_FD_END,
		/* NTV2_FBF_10BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_10BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K*5/16,			AJA_FD_END,	//	Supported in SDK 16.0
		/* NTV2_FBF_8BIT_YCBCR_420PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/4,				AJA_FD_END,
		/* NTV2_FBF_8BIT_YCBCR_422PL2 */		AJA_FD_BEGIN			HD_NUMLINES_4K,					HD_NUMCOMPONENTPIXELS_4K,			HD_NUMCOMPONENTPIXELS_4K/4,				AJA_FD_END
	}
};	//	formatDescriptorTable


NTV2FormatDescriptor::NTV2FormatDescriptor (const NTV2Standard			inStandard,
											const NTV2FrameBufferFormat	inFrameBufferFormat,
											const NTV2VANCMode			inVancMode)
{
	MakeInvalid ();
	if (!NTV2_IS_VALID_STANDARD(inStandard))
		return;	//	bad standard
	if (!NTV2_IS_VALID_FRAME_BUFFER_FORMAT(inFrameBufferFormat))
		return;	//	bad FBF
	if (!NTV2_IS_VALID_VANCMODE(inVancMode))
		return;	//	bad Vanc mode
	if (NTV2_IS_FBF_PLANAR(inFrameBufferFormat) && NTV2_IS_VANCMODE_ON(inVancMode))
		return;	//	can't do VANC mode for planar formats

	//	The 'formatDescriptorTable' handles everything but VANC...
	const NTV2FormatDescriptor	&	result			(formatDescriptorTable[inStandard][inFrameBufferFormat]);
	*this = result;

	mStandard		= inStandard;
	mPixelFormat	= inFrameBufferFormat;
	mVancMode		= inVancMode;
	mFrameGeometry	= ::GetVANCFrameGeometry(::GetGeometryFromStandard(mStandard), mVancMode);
	mNumBitsLuma	= gBitsPerComponent[mPixelFormat][0];
	mNumBitsChroma	= gBitsPerComponent[mPixelFormat][1];
	mNumBitsAlpha	= gBitsPerComponent[mPixelFormat][2];

	//	Account for VANC...
	if (NTV2_IS_VANCMODE_ON(inVancMode))
	{
		const ULWord					numActiveLines	(numLines);
		switch (inStandard)
		{
			case NTV2_STANDARD_2Kx1080p:
			case NTV2_STANDARD_2Kx1080i:
			case NTV2_STANDARD_1080:
			case NTV2_STANDARD_1080p:	numLines = NTV2_IS_VANCMODE_TALLER(inVancMode) ? 1114 : 1112;	break;

			case NTV2_STANDARD_720:		numLines = 740;													break;

			case NTV2_STANDARD_525:		numLines = NTV2_IS_VANCMODE_TALLER(inVancMode) ? 514 : 508;		break;

			case NTV2_STANDARD_625:		numLines = NTV2_IS_VANCMODE_TALLER(inVancMode) ? 612 : 598;		break;

			case NTV2_STANDARD_2K:		numLines = 1588;												break;

			default:					numLines = numActiveLines;										break;
		}
		firstActiveLine = numLines - numActiveLines;
	}

	if (numLines  &&  NTV2_IS_FBF_PLANAR(inFrameBufferFormat))
		FinalizePlanar();

}	//	construct from NTV2Standard & NTV2VANCMode


void NTV2FormatDescriptor::FinalizePlanar (void)
{
	NTV2_ASSERT(NTV2_IS_FBF_PLANAR(mPixelFormat));
	NTV2_ASSERT(numLines);
	switch (mPixelFormat)
	{
		case NTV2_FBF_10BIT_YCBCR_420PL2:
		case NTV2_FBF_10BIT_YCBCR_422PL2:
		case NTV2_FBF_8BIT_YCBCR_420PL2:
		case NTV2_FBF_8BIT_YCBCR_422PL2:
		{
			mNumPlanes = 2;
			mLinePitch[0] = linePitch * 4;
			mLinePitch[1] = linePitch * 4;
			break;
		}

		case NTV2_FBF_8BIT_YCBCR_420PL3:
		case NTV2_FBF_8BIT_YCBCR_422PL3:
		{
			mNumPlanes = 3;
			mLinePitch[0] = numPixels;
			const ULWord	lumaRasterBytes		(numLines * numPixels);
			ULWord			chromaRasterBytes	(lumaRasterBytes / 2);
			if (lumaRasterBytes % 4)
				{MakeInvalid();	break;}
			mLinePitch[1] = mLinePitch[2] = chromaRasterBytes / numLines;	//	For HD1080p: 1920 px/line ==> 1920/4 ==> 480 bytes/line
			if (chromaRasterBytes % numLines)
				MakeInvalid();
			break;
		}

		case NTV2_FBF_10BIT_YCBCR_420PL3_LE:
		case NTV2_FBF_10BIT_YCBCR_422PL3_LE:
		{
			mNumPlanes = 3;
			mLinePitch[0] = numPixels * sizeof(uint16_t);	//	Y is 2-bytes per pixel	//	1080p:  1920 x 2 ==> 3840 bytes per line)
			const ULWord	lumaRasterBytes		(numLines * mLinePitch[0]);				//	1080p:  1080 * 3840 ==> 4,147,200 bytes (0x003F4800)
			ULWord			chromaRasterBytes	(lumaRasterBytes / 2);				//	422 1080p:  2,073,600 bytes (0x001FA400)
			mLinePitch[1] = mLinePitch[2] = chromaRasterBytes / numLines;			//	422 1080p:	1920 bytes/line (0x780)
			if (chromaRasterBytes % numLines)										//	420 1080p:	960 bytes/line (0x3C0)
				MakeInvalid();
			break;
		}

		default:	MakeInvalid();	break;
	}
}

NTV2FormatDescriptor::NTV2FormatDescriptor (const NTV2VideoFormat		inVideoFormat,
											const NTV2FrameBufferFormat	inFrameBufferFormat,
											const NTV2VANCMode			inVancMode)
{
	MakeInvalid ();
	const NTV2Standard	inStandard	(::GetNTV2StandardFromVideoFormat (inVideoFormat));
	if (!NTV2_IS_VALID_STANDARD(inStandard))
		return;	//	bad standard
	if (!NTV2_IS_VALID_FRAME_BUFFER_FORMAT(inFrameBufferFormat))
		return;	//	bad FBF
	if (!NTV2_IS_VALID_VANCMODE(inVancMode))
		return;	//	bad Vanc mode
//	In SDK 16.0, some experimental Corvid88 firmware can do this now:
//	if (NTV2_IS_FBF_PLANAR(inFrameBufferFormat) && NTV2_IS_VANCMODE_ON(inVancMode))
//		return;	//	can't do VANC mode for planar formats

	*this = formatDescriptorTable[inStandard][inFrameBufferFormat];		//	The 'formatDescriptorTable' handles everything but VANC

	mVideoFormat	= inVideoFormat;
	mStandard		= inStandard;
	mPixelFormat	= inFrameBufferFormat;
	mVancMode		= inVancMode;
	mFrameGeometry	= ::GetVANCFrameGeometry(::GetNTV2FrameGeometryFromVideoFormat(mVideoFormat), mVancMode);
	mNumBitsLuma	= gBitsPerComponent[mPixelFormat][0];
	mNumBitsChroma	= gBitsPerComponent[mPixelFormat][1];
	mNumBitsAlpha	= gBitsPerComponent[mPixelFormat][2];

	//	Account for VANC...
	if (NTV2_IS_VANCMODE_ON(inVancMode))
	{
		const ULWord					numActiveLines	(numLines);
		switch (inStandard)
		{
			case NTV2_STANDARD_2Kx1080p:
			case NTV2_STANDARD_2Kx1080i:
			case NTV2_STANDARD_1080:
			case NTV2_STANDARD_1080p:	numLines = NTV2_IS_VANCMODE_TALLER(inVancMode) ? 1114 : 1112;	break;

			case NTV2_STANDARD_720:		numLines = 740;													break;

			case NTV2_STANDARD_525:		numLines = NTV2_IS_VANCMODE_TALLER(inVancMode) ? 514 : 508;		break;

			case NTV2_STANDARD_625:		numLines = NTV2_IS_VANCMODE_TALLER(inVancMode) ? 612 : 598;		break;

			case NTV2_STANDARD_2K:		numLines = 1588;												break;

			default:					numLines = numActiveLines;										break;
		}
		firstActiveLine = numLines - numActiveLines;
	}
	if (NTV2_IS_FBF_PLANAR (inFrameBufferFormat))
		FinalizePlanar();

}	//	construct from NTV2VideoFormat & NTV2VANCMode

NTV2FormatDescriptor::NTV2FormatDescriptor (const NTV2FrameSize &	inFrameSize,
											const NTV2PixelFormat	inPixelFormat,
											const NTV2VANCMode		inVancMode)
{
	MakeInvalid();
	if (!inFrameSize)
		return;	//	bad frame size
	if (!NTV2_IS_VALID_FRAME_BUFFER_FORMAT(inPixelFormat))
		return;	//	bad FBF
	if (!NTV2_IS_VALID_VANCMODE(inVancMode))
		return;	//	bad Vanc mode

	NTV2FrameSize frameSize(inFrameSize);
	mFrameGeometry = frameSize;
	if (NTV2_IS_TALL_VANC_GEOMETRY(mFrameGeometry))
	{
		if (inVancMode != NTV2_VANCMODE_TALL)
			return;	//	vanc mode mismatched
		NTV2FrameGeometry tmp(::GetNormalizedFrameGeometry(mFrameGeometry));
		const NTV2FrameSize tall(mFrameGeometry), normal(tmp);
		firstActiveLine = tall.height() - normal.height();	//	1st active line
		mVancMode = inVancMode;
	}
	else if (NTV2_IS_TALLER_VANC_GEOMETRY(mFrameGeometry))
	{
		if (inVancMode != NTV2_VANCMODE_TALLER)
			return;	//	vanc mode mismatched
		NTV2FrameGeometry tmp(::GetNormalizedFrameGeometry(mFrameGeometry));
		const NTV2FrameSize taller(mFrameGeometry), normal(tmp);
		firstActiveLine = taller.height() - normal.height();	//	1st active line
		mVancMode = inVancMode;
	}
	else if (NTV2_IS_VALID_NTV2FrameGeometry(mFrameGeometry))
	{
		NTV2FrameGeometry tallTaller(::GetVANCFrameGeometry(mFrameGeometry, inVancMode));
		if (!NTV2_IS_VALID_NTV2FrameGeometry(tallTaller))
			return;	//	bad geometry
		if (tallTaller != mFrameGeometry)
		{
			mFrameGeometry = tallTaller;
			frameSize = tallTaller;
			firstActiveLine = frameSize.height() - inFrameSize.height();	//	1st active line
		}
	}
	else if (NTV2_IS_VANCMODE_ON(inVancMode))
		return;	//	vanc enabled but non-standard geometry
	numLines = frameSize.height();
	numPixels = frameSize.width();
	//	mLinePitch[0]:	# bytes per line
	//	linePitch:	# 32-bit words per line -- shadows mLinePitch[0] / sizeof(ULWord)
	mPixelFormat = inPixelFormat;
	switch (mPixelFormat)
	{
		case NTV2_FBF_8BIT_YCBCR:
		case NTV2_FBF_8BIT_YCBCR_YUY2:			mLinePitch[0] = 2*numPixels;	linePitch = mLinePitch[0]/4;	break;

		case NTV2_FBF_10BIT_YCBCR:
		case NTV2_FBF_10BIT_YCBCR_DPX:			mLinePitch[0] = 16*numPixels/6;	linePitch = mLinePitch[0]/4;	break;

		case NTV2_FBF_ARGB:
		case NTV2_FBF_RGBA:
		case NTV2_FBF_ABGR:
		case NTV2_FBF_10BIT_RGB:
		case NTV2_FBF_10BIT_DPX:
		case NTV2_FBF_10BIT_DPX_LE:				mLinePitch[0] = 4*numPixels;	linePitch = mLinePitch[0]/4;	break;

		case NTV2_FBF_24BIT_RGB:
		case NTV2_FBF_24BIT_BGR:				mLinePitch[0] = 3*numPixels;	linePitch = mLinePitch[0]/4;	break;

		case NTV2_FBF_48BIT_RGB:				mLinePitch[0] = 6*numPixels;	linePitch = mLinePitch[0]/4;	break;

		case NTV2_FBF_12BIT_RGB_PACKED:			mLinePitch[0] = 36*numPixels/8;	linePitch = mLinePitch[0]/4;	break;

		case NTV2_FBF_8BIT_YCBCR_420PL3:
		case NTV2_FBF_8BIT_YCBCR_422PL3:		linePitch = numPixels/4;	break;

		case NTV2_FBF_10BIT_YCBCR_420PL3_LE:
		case NTV2_FBF_10BIT_YCBCR_422PL3_LE:	linePitch = numPixels/2;	break;

		case NTV2_FBF_10BIT_YCBCR_420PL2:
		case NTV2_FBF_10BIT_YCBCR_422PL2:		linePitch = 5*numPixels/16;	break;

		default:	mLinePitch[0] = 0;	linePitch = 0;	break;	//	unsupported
	}
	mNumBitsLuma	= gBitsPerComponent[mPixelFormat][0];
	mNumBitsChroma	= gBitsPerComponent[mPixelFormat][1];
	mNumBitsAlpha	= gBitsPerComponent[mPixelFormat][2];
	if (NTV2_IS_FBF_PLANAR(inPixelFormat))
		FinalizePlanar();
}	//	Construct from NTV2FrameSize & NTV2VANCMode


NTV2FormatDescriptor::NTV2FormatDescriptor ()
{
	MakeInvalid();
}


NTV2FormatDescriptor::NTV2FormatDescriptor (const ULWord inNumLines, const ULWord inNumPixels, const ULWord inLinePitch, const ULWord in1stActiveLine,
											const UByte inNumLumaBits, const UByte inNumChromaBits, const UByte inNumAlphaBits)
	:	numLines		(inNumLines),
		numPixels		(inNumPixels),
		linePitch		(inLinePitch),
		firstActiveLine	(in1stActiveLine),
		mStandard		(NTV2_STANDARD_INVALID),
		mVideoFormat	(NTV2_FORMAT_UNKNOWN),
		mPixelFormat	(NTV2_FBF_INVALID),
		mVancMode		(NTV2_VANCMODE_INVALID),
		mNumPlanes		(1),
		mFrameGeometry	(NTV2_FG_INVALID),
		mNumBitsLuma	(inNumLumaBits),
		mNumBitsChroma	(inNumChromaBits),
		mNumBitsAlpha	(inNumAlphaBits)
{
	mLinePitch[0] = inLinePitch * 4;	//	mLinePitch is in bytes, inLinePitch is in 32-bit longwords
	mLinePitch[1] = mLinePitch[2] = mLinePitch[3] = 0;
}


void NTV2FormatDescriptor::MakeInvalid (void)
{
	numLines		= 0;
	numPixels		= 0;
	linePitch		= 0;
	firstActiveLine	= 0;
	mStandard		= NTV2_STANDARD_INVALID;
	mVideoFormat	= NTV2_FORMAT_UNKNOWN;
	mPixelFormat	= NTV2_FBF_INVALID;
	mVancMode		= NTV2_VANCMODE_INVALID;
	mLinePitch[0] = mLinePitch[1] = mLinePitch[2] = mLinePitch[3] = 0;
	mNumPlanes		= 0;
	mFrameGeometry	= NTV2_FG_INVALID;
	mNumBitsLuma	= mNumBitsChroma = mNumBitsAlpha = 0;
}

ULWord NTV2FormatDescriptor::GetTotalBytes (void) const
{
	ULWord bytes(0);
	UWord plane(0);
	do
	{
		bytes += GetTotalRasterBytes(plane);
	} while (++plane < GetNumPlanes());
	return bytes;
}

ULWord NTV2FormatDescriptor::GetVideoWriteSize (ULWord inPageSizeBytes) const
{
	static const ULWord s4K(0x00001000), s64K(0x00010000);
	ULWord result(GetTotalBytes()), pageSizeBytes(inPageSizeBytes);
	if (inPageSizeBytes != s4K)
	{	//	Ensure power-of-2...
		pageSizeBytes = s64K;	//	start at 64K
		do
		{
			if (pageSizeBytes & inPageSizeBytes)
				break;	//	exit at MSB
			pageSizeBytes >>= 1;
		} while (pageSizeBytes > s4K);	//	4K minimum
	}
	if (result % pageSizeBytes)
		result = ((result / pageSizeBytes) + 1) * pageSizeBytes;
	return result;
}

ULWord NTV2FormatDescriptor::GetVerticalSampleRatio (const UWord inPlaneIndex0) const
{
	if (inPlaneIndex0 >= mNumPlanes)
		return 0;	//	no such plane!
	switch (mPixelFormat)
	{
		case NTV2_FBF_8BIT_YCBCR_420PL3:
		case NTV2_FBF_10BIT_YCBCR_420PL3_LE:
		case NTV2_FBF_10BIT_YCBCR_420PL2:
		case NTV2_FBF_8BIT_YCBCR_420PL2:
			return inPlaneIndex0 ? 2 : 1;
		default:
			break;
	}
	return 1;	//	Defaults to 1
}

bool NTV2FormatDescriptor::Is2KFormat (void) const
{
	if (NTV2_IS_VALID_VIDEO_FORMAT(mVideoFormat))
		return NTV2_IS_2K_1080_VIDEO_FORMAT(mVideoFormat);
	else if (NTV2_IS_VALID_STANDARD(mStandard))
		return NTV2_IS_2K1080_STANDARD(mStandard);
	return false;
}


bool NTV2FormatDescriptor::operator == (const NTV2FormatDescriptor & inRHS) const
{
	return numLines == inRHS.numLines  &&  numPixels == inRHS.numPixels
			&&  mNumPlanes == inRHS.mNumPlanes  &&  firstActiveLine == inRHS.firstActiveLine
			&&  mLinePitch[0] == inRHS.mLinePitch[0];
}


string NTV2FormatDescriptor::PlaneToString (const UWord inPlaneIndex0) const
{
	static string	emptyString;
	if (NTV2_IS_VALID_FRAME_BUFFER_FORMAT(mPixelFormat)  &&  inPlaneIndex0 < GetNumPlanes())
		return gPlaneLabels[mPixelFormat][inPlaneIndex0];
	return emptyString;
}


UWord NTV2FormatDescriptor::ByteOffsetToPlane (const ULWord inByteOffset) const
{
	if (!IsPlanar())
		return inByteOffset < GetTotalRasterBytes() ? 0 : 0xFFFF;

	ULWord	byteOffset	(0);
	UWord	plane		(0);
	do
	{
		byteOffset += GetTotalRasterBytes(plane);
		if (inByteOffset < byteOffset)
			return plane;
	} while (++plane < GetNumPlanes());
	return 0xFFFF;
}


UWord NTV2FormatDescriptor::ByteOffsetToRasterLine (const ULWord inByteOffset) const
{
	const UWord	origPlane	(ByteOffsetToPlane(inByteOffset));
	if (origPlane == 0xFFFF)
		return 0xFFFF;
	ULWord	byteOffsetToStartOfPlane (0);
	UWord	plane	(origPlane);
	while (plane)
		byteOffsetToStartOfPlane += GetTotalRasterBytes(--plane);
	NTV2_ASSERT(inByteOffset >= byteOffsetToStartOfPlane);
	const ULWord lineOffset	((inByteOffset - byteOffsetToStartOfPlane) / GetBytesPerRow(origPlane));
	return UWord(lineOffset);
}


bool NTV2FormatDescriptor::IsAtLineStart (ULWord inByteOffset) const
{
	if (!IsValid())
		return false;
	UWord	plane	(ByteOffsetToPlane(inByteOffset));
	if (plane >= GetNumPlanes())
		return false;
	if (plane > 0)
		inByteOffset -= GetTotalRasterBytes(0);
	if (plane > 1)
		inByteOffset -= GetTotalRasterBytes(1);
	return (inByteOffset % GetBytesPerRow(plane)) == 0;
}

bool NTV2FormatDescriptor::GetRowBuffer (const NTV2Buffer & inFrameBuffer, NTV2Buffer & inOutRowBuffer,  const ULWord inRowIndex0,  const UWord inPlaneIndex0) const
{
	inOutRowBuffer.Deallocate();
	if (inRowIndex0 >= numLines)
		return false;	//	Bad row index
	if (inPlaneIndex0 >= mNumPlanes)
		return false;	//	Bad plane index
	if (inFrameBuffer.IsNULL())
		return false;	//	NULL/empty frame buffer

	ULWord byteOffset(0);	//	Set byteOffset to start of desired plane
	if (inPlaneIndex0 > 0)
		byteOffset += GetTotalRasterBytes(0);
	if (inPlaneIndex0 > 1)
		byteOffset += GetTotalRasterBytes(1);
	if (inPlaneIndex0 > 2)
		byteOffset += GetTotalRasterBytes(2);

	ULWord bytesPerRow(GetBytesPerRow(inPlaneIndex0));
	byteOffset += inRowIndex0 * bytesPerRow;	//	Add offset to start of requested line
	if (byteOffset+bytesPerRow > inFrameBuffer.GetByteCount())
		return false;	//	Past end of frame buffer

	return inOutRowBuffer.Set(inFrameBuffer.GetHostAddress(byteOffset), bytesPerRow);
}

const void * NTV2FormatDescriptor::GetRowAddress (const void * pInStartAddress, const ULWord inRowIndex0, const UWord inPlaneIndex0) const
{
	if (inRowIndex0 >= numLines)
		return AJA_NULL;
	if (inPlaneIndex0 >= mNumPlanes)
		return AJA_NULL;
	if (!pInStartAddress)
		return AJA_NULL;

	const UByte * pStart (AsConstUBytePtr(pInStartAddress));
	NTV2_ASSERT(pStart);
	if (inPlaneIndex0 > 0)
		pStart += GetTotalRasterBytes(0);
	if (inPlaneIndex0 > 1)
		pStart += GetTotalRasterBytes(1);
	if (inPlaneIndex0 > 2)
		pStart += GetTotalRasterBytes(2);
	return pStart  +  inRowIndex0 * GetBytesPerRow(inPlaneIndex0);
}


void * NTV2FormatDescriptor::GetWriteableRowAddress (void * pInStartAddress, const ULWord inRowIndex0, const UWord inPlaneIndex0) const
{
	if (inRowIndex0 >= numLines)
		return AJA_NULL;
	if (inPlaneIndex0 >= mNumPlanes)
		return AJA_NULL;

	UByte *	pStart (AsUBytePtr(pInStartAddress));
	if (inPlaneIndex0 > 0)
		pStart += GetTotalRasterBytes(0);
	if (inPlaneIndex0 > 1)
		pStart += GetTotalRasterBytes(1);
	if (inPlaneIndex0 > 2)
		pStart += GetTotalRasterBytes(2);
	return pStart  +  inRowIndex0 * GetBytesPerRow(inPlaneIndex0);
}


ULWord NTV2FormatDescriptor::RasterLineToByteOffset (const ULWord inRowIndex0, const UWord inPlaneIndex0) const
{
	if (inRowIndex0 >= numLines)
		return 0xFFFFFFFF;
	if (inPlaneIndex0 >= mNumPlanes)
		return 0xFFFFFFFF;

	ULWord	offset (0);
	if (inPlaneIndex0 > 0)
		offset += GetTotalRasterBytes(0);
	if (inPlaneIndex0 > 1)
		offset += GetTotalRasterBytes(1);
	if (inPlaneIndex0 > 2)
		offset += GetTotalRasterBytes(2);
	return offset  +  inRowIndex0 * GetBytesPerRow(inPlaneIndex0);
}


bool NTV2FormatDescriptor::GetFirstChangedRow (const void * pInStartAddress1, const void * pInStartAddress2, ULWord & outFirstChangedRowNum) const
{
	outFirstChangedRowNum = 0xFFFFFFFF;
	if (!pInStartAddress1)
		return false;	//	NULL start address 1
	if (!pInStartAddress2)
		return false;	//	NULL start address 2
	if (pInStartAddress1 == pInStartAddress2)
		return false;	//	same start addresses
	if (!IsValid())
		return false;	//	invalid

	UWord	plane	(0);
	do
	{
		for (outFirstChangedRowNum = 0;  outFirstChangedRowNum < GetFullRasterHeight();  outFirstChangedRowNum++)
			if (::memcmp(GetRowAddress(AsConstUBytePtr(pInStartAddress1), outFirstChangedRowNum, plane), GetRowAddress(pInStartAddress2, outFirstChangedRowNum, plane), GetBytesPerRow(plane)))
				return true;	//	CHANGE FOUND -- Success!
	} while (++plane < GetNumPlanes());
	outFirstChangedRowNum = 0xFFFFFFFF;	//	NO CHANGES FOUND
	return true;	//	Success
}


bool NTV2FormatDescriptor::GetChangedLines (NTV2RasterLineOffsets & outDiffs, const void * pInBuffer1, const void * pInBuffer2, const ULWord inMaxLines) const
{
	outDiffs.clear ();
	if (!IsValid())
		return false;	//	Invalid format descriptor
	if (!pInBuffer1 || !pInBuffer2)
		return false;	//	NULL or empty buffers

	const UByte *	pStartAddress1	(AsConstUBytePtr(pInBuffer1));
	const UByte *	pStartAddress2	(AsConstUBytePtr(pInBuffer2));

	if (!GetRowAddress (pStartAddress1, GetFullRasterHeight() - 1))
		return false;	//	buffer too small to fetch given rowOffset
	if (pInBuffer1 == pInBuffer2)
		return true;	//	Same buffer -- no diffs (success!)
	ULWord	maxLines	(inMaxLines ? inMaxLines : GetFullRasterHeight());
	if (maxLines > GetFullRasterHeight())
		maxLines = GetFullRasterHeight();	//	Clamp maxLines to full raster height

	UWord	plane	(0);
	do
	{
		for (ULWord rowOffset(0);  rowOffset < maxLines;  rowOffset++)
			if (::memcmp(GetRowAddress(pStartAddress1, rowOffset, plane), GetRowAddress(pStartAddress2, rowOffset, plane), GetBytesPerRow(plane)))
				outDiffs.push_back(rowOffset);
	} while (++plane < GetNumPlanes());
	return true;
}


ostream & NTV2FormatDescriptor::Print (ostream & oss, const bool inDetailed) const
{
	UWord	plane	(0);
	if (!IsValid ())
		oss << "INVALID: ";
	oss	<< "lines=" << DEC(GetFullRasterHeight())
		<< " px/line=" << DEC(GetRasterWidth());
	do
	{
		if (IsPlanar())
			oss << " PL" << plane << ":";
		oss << " bytes/line=" << DEC(GetBytesPerRow(plane));
	} while (++plane < GetNumPlanes());
	if (firstActiveLine)
		oss << " 1stAct=" << DEC(firstActiveLine);
	if (inDetailed)
	{
		plane = 0;
		if (IsPlanar())
			do
			{
				oss << " PL" << plane << " bytes=" << xHEX0N(GetTotalRasterBytes(plane),8);
			} while (++plane < GetNumPlanes());
		else
			oss << " bytes=" << xHEX0N(GetTotalRasterBytes(),8);
		if (NTV2_IS_VALID_VIDEO_FORMAT(mVideoFormat))
			oss << " " << ::NTV2VideoFormatToString(mVideoFormat);
		else
			oss << " " << ::NTV2StandardToString(mStandard) << (Is2KFormat() ? " 2K" : "");
		if (NTV2_IS_VANCMODE_ON(mVancMode))
			oss << (NTV2_IS_VANCMODE_TALLER(mVancMode) ? " TallerVANC" : " TallVANC");
		if (NTV2_IS_VALID_FRAME_BUFFER_FORMAT(mPixelFormat))
			oss << " " << ::NTV2FrameBufferFormatToString(mPixelFormat);
		if (IsRGB())
			oss << " bitsC" << DEC(UWord(GetNumBitsChroma()));
		else
			oss << " bitsY" << DEC(UWord(GetNumBitsLuma()));
		if (HasAlpha())
			oss << "A" << DEC(UWord(GetNumBitsAlpha()));
	}
	return oss;
}


//	SMPTE LINE NUMBERS:

//	NTV2_VANCMODE_OFF							1080i	720p	525i	625i	1080p	2K		2K1080p		2K1080i		3840x2160p	4096x2160p	3840HFR		4096HFR		7680x4320	8192x4320
static const ULWord	LineNumbersF1 []	=	{	21,		26,		21,		23,		42,		211,	42,			21,			42,			42,			42,			42,			42,			42,			0	};
static const ULWord	LineNumbersF2 []	=	{	584,	27,		283,	336,	43,		1201,	43,			584,		43,			43,			43,			43,			43,			43,			0	};
//	NTV2_VANCMODE_TALL
static const ULWord	LineNumbersF1t []	=	{	5,		6,		10,		12,		10,		211,	10,			5,			10,			10,			10,			10,			10,			10,			0	};
static const ULWord	LineNumbersF2t []	=	{	568,	7,		272,	325,	11,		1201,	11,			568,		11,			11,			11,			11,			11,			11,			0	};
//	NTV2_VANCMODE_TALLER
static const ULWord	LineNumbersF1tt []	=	{	4,		6,		7,		5,		8,		211,	8,			4,			8,			8,			8,			8,			8,			8,			0	};
static const ULWord	LineNumbersF2tt []	=	{	567,	7,		269,	318,	9,		1201,	9,			567,		9,			9,			9,			9,			9,			9,			0	};
																																										                    
static const ULWord	LineNumbersF1Last[]	=	{	560,	745,	263,	310,	1121,	0,		1121,		0,			2201,		2201,		2201,		2201,		4361,		4361,		0	};
static const ULWord	LineNumbersF2Last[]	=	{	1123,	745,	525,	623,	1121,	0,		1121,		0,			0,			0,			0,			0,			0,			0,			0	};


bool NTV2FormatDescriptor::GetSMPTELineNumber (const ULWord inLineOffset, ULWord & outSMPTELine, bool & outIsField2) const
{
	outIsField2 = false;
	outSMPTELine = 0;
	if (!IsValid())
		return false;
	if (!NTV2_IS_VALID_STANDARD(mStandard))
		return false;
	if (!NTV2_IS_VALID_VANCMODE(mVancMode))
		return false;
	if (inLineOffset >= GetFullRasterHeight())
		return false;

	const bool		is525i		(mStandard == NTV2_STANDARD_525);
	if (!NTV2_IS_PROGRESSIVE_STANDARD(mStandard))
		outIsField2 = (inLineOffset & 1)  ?  !is525i  :  is525i;

	const ULWord	divisor		(NTV2_IS_PROGRESSIVE_STANDARD(mStandard)  ?  1  :  2);
	ULWord			smpteLine	(0);

	switch (mVancMode)
	{
		case NTV2_VANCMODE_OFF:		smpteLine =  outIsField2  ?  LineNumbersF2[mStandard]   :  LineNumbersF1[mStandard];	break;
		case NTV2_VANCMODE_TALL:	smpteLine =  outIsField2  ?  LineNumbersF2t[mStandard]  :  LineNumbersF1t[mStandard];	break;
		case NTV2_VANCMODE_TALLER:	smpteLine =  outIsField2  ?  LineNumbersF2tt[mStandard] :  LineNumbersF1tt[mStandard];	break;
		default:					break;
	}
	outSMPTELine = inLineOffset/divisor + smpteLine;
	return true;
}


bool NTV2FormatDescriptor::GetLineOffsetFromSMPTELine (const ULWord inSMPTELine, ULWord & outLineOffset) const
{
	outLineOffset = 0xFFFFFFFF;
	if (!IsValid())
		return false;
	if (!NTV2_IS_VALID_STANDARD(mStandard))
		return false;
	if (!NTV2_IS_VALID_VANCMODE(mVancMode))
		return false;

	ULWord			firstF1Line(0),	firstF2Line(0);
	const ULWord	lastF1Line	(LineNumbersF1Last[mStandard]);
	const ULWord	lastF2Line	(LineNumbersF2Last[mStandard]);
	const ULWord	is525i		(mStandard == NTV2_STANDARD_525  ?  1  :  0);
	const ULWord	not525i		(mStandard != NTV2_STANDARD_525  ?  1  :  0);

	switch (mVancMode)
	{
		case NTV2_VANCMODE_OFF:		firstF1Line = LineNumbersF1[mStandard];		firstF2Line = LineNumbersF2[mStandard];		break;
		case NTV2_VANCMODE_TALL:	firstF1Line = LineNumbersF1t[mStandard];	firstF2Line = LineNumbersF2t[mStandard];	break;
		case NTV2_VANCMODE_TALLER:	firstF1Line = LineNumbersF1tt[mStandard];	firstF2Line = LineNumbersF2tt[mStandard];	break;
		case NTV2_VANCMODE_INVALID:	return false;
	}
	if (inSMPTELine < firstF1Line)
		return false;	//	Above first line in FB raster
	if (NTV2_IS_PROGRESSIVE_STANDARD(mStandard)  &&  lastF1Line  &&  inSMPTELine > lastF1Line)
		return false;	//	Past last line
	if (!NTV2_IS_PROGRESSIVE_STANDARD(mStandard)  &&  lastF1Line  &&  inSMPTELine < firstF2Line  &&  inSMPTELine > lastF1Line)
		return false;	//	In the void past end of F1 and above F2 first line
	if (!NTV2_IS_PROGRESSIVE_STANDARD(mStandard)  &&  lastF2Line  &&  inSMPTELine > lastF2Line)
		return false;	//	Past last line

	if (NTV2_IS_PROGRESSIVE_STANDARD(mStandard))
		outLineOffset = inSMPTELine - firstF1Line;
	else if (inSMPTELine >= firstF2Line)
		outLineOffset = (inSMPTELine - firstF2Line) * 2 + not525i;
	else
		outLineOffset = (inSMPTELine - firstF1Line) * 2 + is525i;
	return outLineOffset < GetFullRasterHeight();
}


ostream & NTV2FormatDescriptor::PrintSMPTELineNumber (ostream & inOutStream, const ULWord inLineOffset, const bool inForTextMode) const
{
	ULWord	smpteLine	(0);
	bool	isF2		(false);
	if (GetSMPTELineNumber (inLineOffset, smpteLine, isF2))
	{
		if (!NTV2_IS_PROGRESSIVE_STANDARD (mStandard))
			inOutStream << "F" << (isF2 ? "2" : "1") << (inForTextMode ? "" : " ");
		if (inForTextMode)
			inOutStream << "L" << DEC0N(smpteLine,4);
		else
			inOutStream << "L" << DEC(smpteLine);	//	(inLineOffset/divisor + smpteLine);
	}
	return inOutStream;
}

NTV2FrameDimensions NTV2FormatDescriptor::GetFullRasterDimensions (void) const
{
	return NTV2FrameDimensions (GetRasterWidth(), GetRasterHeight(false));
}

NTV2FrameDimensions NTV2FormatDescriptor::GetVisibleRasterDimensions (void) const
{
	return NTV2FrameDimensions (GetRasterWidth(), GetRasterHeight(true));
}


NTV2SegmentedXferInfo & NTV2FormatDescriptor::GetSegmentedXferInfo (NTV2SegmentedXferInfo & inSegmentInfo, const bool inIsSource) const
{
	if (IsValid())
	{	//	Full visible raster
		inSegmentInfo.setElementLength(1).setSegmentCount(GetRasterHeight(true)).setSegmentLength(GetBytesPerRow());
		if (inIsSource)
			return inSegmentInfo.setSourceOffset(GetBytesPerRow() * GetFirstActiveLine())
								.setSourcePitch(GetBytesPerRow());
		else
			return inSegmentInfo.setDestOffset(GetBytesPerRow() * GetFirstActiveLine())
								.setDestPitch(GetBytesPerRow());
	}
	inSegmentInfo = NTV2SegmentedXferInfo();
	return inSegmentInfo;	
}


//	Q:	WHY IS NTV2SmpteLineNumber's CONSTRUCTOR & GetLastLine IMPLEMENTATION HERE?
//	A:	TO USE THE SAME LineNumbersF1/LineNumbersF2 TABLES (above)

NTV2SmpteLineNumber::NTV2SmpteLineNumber (const NTV2Standard inStandard)
{
	NTV2_ASSERT (inStandard < sizeof(LineNumbersF1) / sizeof(ULWord));
	*this = NTV2SmpteLineNumber (LineNumbersF1[inStandard],  LineNumbersF2[inStandard],  inStandard != NTV2_STANDARD_525,  inStandard);
}


ULWord NTV2SmpteLineNumber::GetLastLine (const NTV2FieldID inRasterFieldID) const
{
	if (!NTV2_IS_VALID_FIELD(inRasterFieldID))
		return 0;

	if (inRasterFieldID == NTV2_FIELD0)
		return firstFieldTop ? LineNumbersF1Last[mStandard] : LineNumbersF2Last[mStandard];
	else
		return firstFieldTop ? LineNumbersF2Last[mStandard] : LineNumbersF1Last[mStandard];
}


ostream & NTV2PrintRasterLineOffsets(const NTV2RasterLineOffsets & inObj, ostream & inOutStream)
{
	NTV2StringList	pieces;
	NTV2RasterLineOffsetsConstIter	iter (inObj.begin());
	ULWord	current		(0xFFFFFFFF);
	ULWord	previous	(0xFFFFFFFF);
	ULWord	first		(0xFFFFFFFF);
	ULWord	last		(0xFFFFFFFF);

	#if 0
		//	Verify sorted ascending...
		current = 0;
		while (iter != inObj.end())
		{
			NTV2_ASSERT (current < previous);
			previous = current;
		}
		iter = inObj.begin();
	#endif	//	_DEBUG

	while (iter != inObj.end())
	{
		current = *iter;
		if (previous == 0xFFFFFFFF)
			previous = first = last = current;	//	First time -- always start new sequence
		else if (current == (previous + 1))
			last = previous = current;			//	Continue sequence
		else if (current == previous)
			;
		else
		{
			ostringstream	oss;
			if (first == last)
				oss << first;
			else
				oss << first << "-" << last;
			pieces.push_back (oss.str ());

			first = last = previous = current;	//	Start new sequence...
		}	//	else sequence break
		++iter;
	}

	if (first != 0xFFFFFFFF && last != 0xFFFFFFFF)
	{
		ostringstream	oss;
		if (first == last)
			oss << first;
		else
			oss << first << "-" << last;
		pieces.push_back (oss.str ());
	}

	for (NTV2StringListConstIter it (pieces.begin());  ; )
	{
		inOutStream << *it;
		if (++it != pieces.end())
			inOutStream << ",";
		else
			break;
	}
	return inOutStream;
}
