/* --------------------------------------------------------------------------
 * module_pmac.c
 * code for test module
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * -------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include "systems.h"

#ifdef WITH_MODULE_PMAC

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include "pbbinput.h"
#include <sys/ioctl.h>
#include <linux/pmu.h>
#include <linux/adb.h>

#include <glib.h>
#include <pbb.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "config_manager.h"
#include "module_pmac.h"
#include "support.h"
#ifdef WITH_PMUD
#  include "tcp.h"
#endif
#include "debug.h"

#define TRACKPAD_PAUSETIME	6	/* time in 1/100 of a second */

struct moddata_pmac {
	char *pmudev;    /* pathname of the PMU device */
	int fd_pmu;      /* filehandle of the PMU device */
	char *adbdev;    /* pathname of the ADB device */
	int fd_adb;      /* filehandle of ADB device */
	char *i2cdev;    /* pathname of the I2C device with ambient sensor and KBD illu */

	int version;     /* PMU version */
	char *identity;  /* Identity string of this laptop */
	int machine;     /* OpenFirmware machine code */
	int lmuaddr;	 /* i2c address of LMU Controller */
	struct modflags_pmac flags;
	int flagschanged;        /* PMU flags that have changed recently */
	int oharevolbutton;      /* level of volume button on OHARE PBs scaled to 0..100 */
	int oharebrightbutton;   /* level of brightness button on OHARE PBs scaled to 0..15 */

	int charge[MAX_BATTERIES];     /* current battery charge */
	int chargemax[MAX_BATTERIES];  /* max battery charge */
	int current[MAX_BATTERIES];    /* electrical currrent out of the batteries */
	int voltage[MAX_BATTERIES];    /* current battery voltage */
	int timeleft;           /* time until battery runs out of fuel. This value is
                             low pass filtered to reduce fast signal fluctuation
                             This filter doesn't reduce accuracy. */

	unsigned short keytpmodeup;     /* trackpad control values */
	unsigned short modtpmodeup;
	unsigned short keytpmodedn;
	unsigned short modtpmodedn;
	int trackpad_mode;
	int trackpad_pause;            /* trackpad switch-off timeout */
	int keyboard_mode;             /* keyboard mode for recent keyboards (0xC[0..4]) */
	int batlog_mode;
	int batlog_cycle;
	int lcdfeedback;               /* correction factor for ambient light sensor */
#if defined(DEBUG) && SIMUAMBIENT
	int ambient;
#endif
} modbase_pmac;

int
pmac_init ()
{
	struct moddata_pmac *base = &modbase_pmac;
	static char devbuffer_pmu[STDBUFFERLEN];
	static char devbuffer_adb[STDBUFFERLEN];
	static char devbuffer_i2c[STDBUFFERLEN];
	static char identitystring[STDBUFFERLEN];
	int sid;
	
	bzero (base, sizeof (struct moddata_pmac));
	bzero (devbuffer_pmu, STDBUFFERLEN);
	bzero (devbuffer_adb, STDBUFFERLEN);
	bzero (devbuffer_i2c, STDBUFFERLEN);
	bzero (identitystring, STDBUFFERLEN);
	base->pmudev   = devbuffer_pmu;
	base->adbdev   = devbuffer_adb;
	base->i2cdev   = devbuffer_i2c;
	base->identity = identitystring;

	if ((sid = registerCfgSection ("MODULE PMAC")) == 0) {
		print_msg (PBB_ERR, _("Can't register configuration section %s, out of memory."), "MODULE PMAC");
		return E_NOMEM;
	} else {
		registerCfgOptionString (sid, "dev_PMU", TAG_PMUDEVICE, 0,
				NULL);
		registerCfgOptionString (sid, "dev_ADB", TAG_ADBDEVICE, 0,
				NULL);
		registerCfgOptionKey (sid, "TPModeUpKey", TAG_TPMODEUPKEY, TAG_TPMODEUPMOD, 0,
				NULL);
		registerCfgOptionKey (sid, "TPModeDownKey", TAG_TPMODEDOWNKEY, TAG_TPMODEDOWNMOD, 0,
				NULL);
		registerCfgOptionList (sid, "TPMode", TAG_TPMODE, 0,
				N_("'"TRACKPAD_NOTAP_NAME"', '"TRACKPAD_TAP_NAME"', '"TRACKPAD_DRAG_NAME"' or '"TRACKPAD_LOCK_NAME"'"),
				TRACKPAD_NOTAP_NAME, TRACKPAD_TAP_NAME, TRACKPAD_DRAG_NAME, TRACKPAD_LOCK_NAME, NULL);
		registerCfgOptionList (sid, "KBDMode", TAG_KBDMODE, 0,
				N_("'"KEYBOARD_FNDISABLED_NAME"', '"KEYBOARD_FNBACK_NAME"' or '"KEYBOARD_FNTOP_NAME"'"),
				KEYBOARD_FNDISABLED_NAME, KEYBOARD_FNBACK_NAME, KEYBOARD_FNTOP_NAME, NULL);
		registerCfgOptionList (sid, "Batlog", TAG_BATLOG, 0,
				N_("'"BATLOG_NONE_NAME"', '"BATLOG_CYCLE_NAME"' or '"BATLOG_LOG_NAME"'"),
				BATLOG_NONE_NAME, BATLOG_CYCLE_NAME, BATLOG_LOG_NAME, NULL);
		registerCfgOptionBool (sid, "NoTapTyping", TAG_NOTAPTYPING, 0,
				N_("switch trackpad to 'notap-mode' while typing"));
	}

	register_function (QUERYQUEUE, pmac_query);
	register_function (CONFIGQUEUE, pmac_configure);
	return 0;
}


int
pmac_open (struct tagitem *taglist)
{
	struct moddata_pmac *base = &modbase_pmac;
	char *devname;
	long DUMMY = 0;
	int val, rc;

	base->fd_pmu		    = -1;
	base->fd_adb		     = -1;
	base->flags.coveropen     = 1;         /* lid is open */
	base->flags.sleepsupported = 0;        /* sleep not supported by default */
	base->flags.set_kbdmode  = 0;          /*    request flags for delayed actions */
	base->flags.goto_sleep   = 0;          /*  /  */
	base->flags.notap_typing = 0;

	base->keytpmodeup	= KEY_BRIGHTNESSUP;
	base->modtpmodeup	= MOD_ALT;
	base->keytpmodedn	= KEY_BRIGHTNESSDOWN;
	base->modtpmodedn	= MOD_ALT;
	base->trackpad_mode	= -1;
	base->trackpad_pause = 0;
	base->keyboard_mode	= -1;
	base->batlog_mode	= BATLOG_NONE;  /* no batlog */
	base->batlog_cycle	= -1;         /* cycle invalid */
	base->timeleft      = -1;
	base->machine       =  0;
#if defined(DEBUG) && SIMUAMBIENT
	base->ambient       = 0;
#endif
#ifdef WITH_PMUD
	print_msg (PBB_INFO, _("pmud support compiled in and active.\n"));
#endif

	devname = (char *) tagfind (taglist, TAG_PMUDEVICE, (long) DEFAULT_PMU);
	if ((rc = copy_path (devname, base->pmudev, TYPE_CHARDEV, CPFLG_NONE)) == 0) {
		if ((base->fd_pmu = open(base->pmudev, O_RDWR)) < 0) {
			print_msg (PBB_ERR, _("Can't open PMU device %s: %s\n"), devname, strerror(errno));
			return E_OPEN;
		}
	} else return rc;

	devname = (char *) tagfind (taglist, TAG_ADBDEVICE, (long) DEFAULT_ADB);
	if ((rc = copy_path (devname, base->adbdev, TYPE_CHARDEV, CPFLG_NONE)) == 0) {
		if ((base->fd_adb = open(base->adbdev, O_RDWR)) < 0) {
			print_msg (PBB_ERR, _("Can't open ADB device %s: %s\n"), devname, strerror(errno));
			return E_OPEN;
		}
	} else return rc;

	base->version = get_pmu_version(base->fd_adb);
	base->machine = getMachineID (base->version);
	snprintf(base->identity, STDBUFFERLEN, "%s - PMU version: %d",
			getMachineName(base->machine), base->version);
	
	ioctl(base->fd_pmu, PMU_IOC_CAN_SLEEP, &val);
	base->flags.sleepsupported = val == 1 ? 1 : 0; /* check if sleep is supported on this system */
	
	/* If this PowerBook has an ambient light sensor connected
	 * to the I2C bus, search for the device and save  name and
	 * bus address
	 */
	if ((base->lmuaddr = getLMUAddress()) != 0)
		strncpy (base->i2cdev, findLMUDevice(base->lmuaddr), STDBUFFERLEN);

	/* If we have an Ambient light sensor in this PowerBook, we
	 * must measure the LCD feedback to this sensor in current
	 * state. To do this we switch the backlight off for a moment
	 * to measure the real ambient light level.
	 */
	if (haveI2CAmbient() || havePMUAmbient()) {  
		print_msg (PBB_INFO, _("LMU found, ambient light sensor and keyboard illumination active.\n"));
		val = process_queue_single (QUERYQUEUE, TAG_HW_BACKLIGHT, -1);
		pmac_set_lcdbacklight (0);    /* measure LCD feedback */
		pmac_set_lcdbacklight (val);  /* restore LCD backlight level */
	}

	pmac_update_flags ();
	pmac_update_batteryinfo ();
	base->flagschanged = 0;

#ifdef DEBUG
	print_msg (PBB_INFO, "DBG: Machine:  %0x\n", base->machine);	
    print_msg (PBB_INFO, "DBG: Keyboard: %s\n",
			haveADBKeyboard () ? "ADB" : "USB");
    print_msg (PBB_INFO, "DBG: Trackpad: %s\n",
			haveADBTrackpad () ? "ADB" : "USB");
    print_msg (PBB_INFO, "DBG: Ambient:  %s\n",
			haveI2CAmbient () ? "I2C" : havePMUAmbient () ? "PMU" : "None");
#endif

	base->trackpad_mode = trackpad_get_config(); /* get trackpad mode */

	if (base->version != OHARE_PMU) {
		base->keyboard_mode = keyboard_get_config();	/* get keyboard config*/

		/* If the kernel has been compiled with CONFIG_PMAC_BACKLIGHT_LEGACY
		 * the following ioctl is needed to disable the ancient backlight
		 * control of the kernel. Otherwise the new sysfs interface to the
		 * backlight will not work correctly on PowerBooks. 
		 * The ioctl doesn't matter if the PMU backlight driver is used or
		 * the kernel legacy code was not compiled in.
		 * Because this is a PowerBook speciality the ioctl is called here and
		 * not in the sysfs backlight driver.
		 */
		ioctl(base->fd_pmu, PMU_IOC_GRAB_BACKLIGHT, &DUMMY); 
	}

	if ((addInputSource (base->fd_pmu, pmac_pmu_handler, NULL, FALSE)) == NULL)
		print_msg (PBB_WARN, _("Can't install PMU input handler. Some functionality may be missing.\n"));

	/* register keyboard handler for trackpad control only if we have an ADB trackpad */
	if (haveADBTrackpad ())
		register_function (KBDQUEUE, pmac_keyboard);
	
	register_function (MOUSEQUEUE, pmac_mouse);
	register_function (T100QUEUE, pmac_timer100);
	register_function (T1000QUEUE, pmac_timer1000);
	register_function (SECUREQUEUE, pmac_secure);
	return 0;
}

int
pmac_close ()
{
	struct moddata_pmac *base = &modbase_pmac;

	batlog_save();            /* save batlog info to disc */
	if (base->fd_pmu != -1)
		close (base->fd_pmu); /* close PMU device */
	if (base->fd_adb != -1)
		close (base->fd_adb); /* close ADB device */

	return 0;
}

int
pmac_exit ()
{
	return 0;
}

void
pmac_keyboard (struct tagitem *taglist)
{
	struct moddata_pmac *base = &modbase_pmac;
	int code, value, mod;

	code = (int) tagfind (taglist, TAG_KEYCODE, 0);
	value = (int) tagfind (taglist, TAG_KEYREPEAT, 0);
	mod = (int) tagfind (taglist, TAG_MODIFIER, 0);

	if (value == 1 || value == 2) {
		/* trackpad tapping control
		 * the left mouse button must be filtered out because it will
		 * break NoTapTyping. The Trackpad will emulate a left mouse
		 * button which causes that 'tapping' will be disabled. The
		 * event system received a 'mouse button down' but a 'mouse
		 * button up' will never be seen. This leads to a weird
		 * trackpad behaviour.
		 */
		if (code != BTN_LEFT && !isModifier (code, mod) && base->flags.notap_typing) {
			if (base->trackpad_pause == 0) {
				trackpad_set_config(TRACKPAD_NOTAP);
#if defined(DEBUG) && TRACKPAD
				printf ("Trackpad tapping disabled\r");
				fflush (stdout);
#endif	
			}
			base->trackpad_pause = TRACKPAD_PAUSETIME;
		}
	}
	
	if (value == 1) {
		/* trackpad mode keys */
		if ((code == base->keytpmodeup) && (mod == base->modtpmodeup)) {
			if (++base->trackpad_mode > TRACKPAD_LAST)
				base->trackpad_mode = 0;
		} else if ((code == base->keytpmodedn) && (mod == base->modtpmodedn)) {
			if (--base->trackpad_mode < 0)
				base->trackpad_mode = TRACKPAD_LAST;
		} else return;

		trackpad_set_config(base->trackpad_mode);
		singletag_to_clients (CHANGEVALUE, TAG_TPMODE, base->trackpad_mode);
	}
}

void
pmac_mouse (struct tagitem *taglist)
{
	struct moddata_pmac *base = &modbase_pmac;

	/* trackpad tapping control */
	if (base->trackpad_pause) {
		base->trackpad_pause = 0;
		trackpad_set_config(base->trackpad_mode);
#if defined(DEBUG) && TRACKPAD
		printf ("Trackpad tapping enabled \r");
		fflush (stdout);
#endif	
	}
}

/* This function is called when the user is idle. That means no key is pressed
 */
void
pmac_secure (struct tagitem *taglist)
{
	struct moddata_pmac *base = &modbase_pmac;
	
	if (base->flags.set_kbdmode == 1) {
		base->flags.set_kbdmode = 0;
		keyboard_set_config(base->keyboard_mode);
	} else if (base->flags.goto_sleep == 1) {
		base->flags.goto_sleep = 0;
		if (base->flags.sleepsupported) {
			batlog_save();
			process_queue_single (CONFIGQUEUE, TAG_PREPAREFORSLEEP, 0);
#ifdef WITH_PMUD
			activate_sleepmode ();
			base->flagschanged = pmac_update_flags ();
#else
			do {
			    activate_sleepmode ();
			    base->flagschanged = pmac_update_flags ();
			} while (base->flags.coveropen == 0);
#endif
			base->timeleft = 7200;       /* reset time value so that the */
				/* filter approximate the real value from top. Otherwise */
				/* an errornous battery warnlevel could be sent to       */
				/* clients because the filtered value doesn't cross the */
				/* thresholds quick enough. This problem occours only  */
				/* after wakeup on battery after the battery had been */
				/* recharged during sleep.                           */
			pmac_update_batteryinfo ();
			process_queue_single (CONFIGQUEUE, TAG_WAKEUPFROMSLEEP, 0);
			base->batlog_cycle = batlog_setup();
		}
	}
}

void
pmac_query (struct tagitem *taglist)
{
	pmac_handle_tags (MODE_QUERY, taglist);
}

void
pmac_configure (struct tagitem *taglist)
{
	pmac_handle_tags (MODE_CONFIG, taglist);
}

void
pmac_handle_tags (int cfgure, struct tagitem *taglist)
{
	struct moddata_pmac *base = &modbase_pmac;
	int err;

	while (taglist->tag != TAG_END) {
		switch (taglist->tag) {
		case TAG_REINIT:      /* system tag */
			trackpad_set_config(base->trackpad_mode);
			base->flags.set_kbdmode = 1;     /* reset keyboard mode */
			break;
		case TAG_PMUDEVICE:
			if (cfgure) {
				if ((err = copy_path ((char *) taglist->data, base->pmudev, TYPE_CHARDEV, CPFLG_NONE)))
					tagerror (taglist, err);
			} else
				taglist->data = (long) base->pmudev;
			break;
		case TAG_ADBDEVICE:
			if (cfgure) {
				if ((err = copy_path ((char *) taglist->data, base->adbdev, TYPE_CHARDEV, CPFLG_NONE)))
					tagerror (taglist, err);
			} else
				taglist->data = (long) base->adbdev;
			break;
		case TAG_BACKLIGHTLEVEL:   /* private tag */
			if (cfgure)	pmac_set_lcdbacklight (taglist->data);
			else		taglist->data = process_queue_single (QUERYQUEUE, TAG_HW_BACKLIGHT, -1);
			break;
		case TAG_TPMODEUPKEY:
			if (cfgure)	base->keytpmodeup = taglist->data;
			else		taglist->data = base->keytpmodeup;
			break;
		case TAG_TPMODEUPMOD:
			if (cfgure)	base->modtpmodeup = taglist->data;
			else		taglist->data = base->modtpmodeup;
			break;
		case TAG_TPMODEDOWNKEY:
			if (cfgure)	base->keytpmodedn = taglist->data;
			else		taglist->data = base->keytpmodedn;
			break;
		case TAG_TPMODEDOWNMOD:
			if (cfgure)	base->modtpmodedn = taglist->data;
			else		taglist->data = base->modtpmodedn;
			break;
		case TAG_TPMODE:
			if (cfgure)	{
				base->trackpad_mode = taglist->data > TRACKPAD_LAST ? 0 : taglist->data;
				trackpad_set_config(base->trackpad_mode);
				singletag_to_clients (CHANGEVALUE, TAG_TPMODE, base->trackpad_mode);
			} else
				taglist->data = base->trackpad_mode;
			break;
		case TAG_NOTAPTYPING:
			if (cfgure)	base->flags.notap_typing = taglist->data & 1;
			else		taglist->data = base->flags.notap_typing;
			break;
		case TAG_KBDMODE:
			if (cfgure) {
				if (base->version != OHARE_PMU) {
					base->keyboard_mode = taglist->data > KEYBOARD_LAST ? 0 : taglist->data;
					if (haveADBKeyboard() && !taglist->data) {
						print_msg (PBB_WARN, _("ADB keyboard can't disable fnmode, force mode to 'fkeyslast'.\n"));
						base->keyboard_mode = KEYBOARD_FNBACK;
					}
					base->flags.set_kbdmode = 1;     /* request a secure function call */
				}
			} else
				taglist->data = base->keyboard_mode;
			break;
		case TAG_BATLOG:
			if (cfgure) {
				batlog_save();
				base->batlog_mode = taglist->data > BATLOG_LAST ? 0 : taglist->data;
				base->batlog_cycle = batlog_setup();		
			} else
				taglist->data = base->batlog_mode;
			break;
		case TAG_BATCYCLE:
			if (cfgure)	tagerror(taglist, E_NOWRITE);
			else		taglist->data = base->batlog_cycle;
			break;
		case TAG_REQUESTSLEEP:    /* private tag */
			if (cfgure)	base->flags.goto_sleep = 1;   /* trigger delayed sleep */
			break;
		case TAG_IDENTITY:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else		taglist->data = (long) base->identity;
			break;
		case TAG_SLEEPSUPPORTED:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else		taglist->data = base->flags.sleepsupported;
			break;
#ifndef WITH_IBAM
		case TAG_TIMEREMAINING:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else		taglist->data = base->timeleft;
			break;
#endif
		case TAG_POWERSOURCE:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else		taglist->data = base->flags.ac_power;
			break;
		case TAG_BATTERYPRESENT:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			/* /proc/pmu is not reliable on machines with removable batteries.
			 * If no batteries were plugged in, /proc/pmu would not be updated
			 * anymore and contains the last battery data, which is not valid
			 * anymore.
			 * So pbbuttonsd ask the PMU for the existance of any battery and
			 * report the values from /proc/pmu only if the answer was yes.
			 */
			else 		taglist->data = base->flags.batpresent ? base->flags.bat_present : 0;
			break;

		case TAG_AMBIENTLIGHT:
			if (!haveI2CAmbient() && !havePMUAmbient())
				tagerror (taglist, E_NOSUPPORT);
			else if (cfgure)
#if defined(DEBUG) && SIMUAMBIENT
				base->ambient = taglist->data;
#else
				tagerror (taglist, E_NOWRITE);
#endif
			else
				taglist->data = getAmbient();
			break;
		case TAG_KEYBLIGHTLEVEL:   /* private tag */
			if (cfgure)
				setKBDIllumination (taglist->data);
			break;
		case TAG_KEYBLIGHTMAX:     /* private tag */
			if (cfgure)	;
			else		taglist->data = KEYBLIGHTMAX;
			break;
		case TAG_SYSINFOSLAVE:
			if (cfgure) ;
			else {
				taglist->data |= haveADBKeyboard () ? 0 : SYSINFO_PB_USB_KBD;
				taglist->data |= haveADBTrackpad () ? 0 : SYSINFO_PB_USB_TPAD;
			}
		}
		taglist++;
	}
}

gboolean
pmac_pmu_handler (int fd, gpointer user_data)
{
	unsigned char intr[16];
	struct tagitem taglist[4];
	int n;
	static int PBpressed = 0;
#if defined(DEBUG) && PMUINTR
	int i;
#endif
	if ((n = read(fd, intr, sizeof(intr))) > 0) {
		switch (intr[0]) {
		case 4:
			if (n < 3 || intr[2] < 1 || intr[2] > 2)
				break;
			call_script ("/sbin/cardctl %s %d", "eject", intr[2]-1);
			break;
		case 0x40:  /* timer interrupt */
			/* no battery present -> no timer interrupts on G3 Pismo*/
			/* n = 2 && intr[1] = 0x04 = %00100 std timer interrupt */
			/* n = 2 && intr[1] = 0x14 = %10100 if a battery is present */
			/* n = 2 && intr[1] = 0x0c = %01100 power button on mac-mini */
			/* n = 6 && intr[1] = 0x1c = %11100 if power button is pressed */
			/* n = 6 && intr[1] = 0x15 = %10101 after cover close */
			if (n == 6 && ((intr[1] >> 3) & 1) != PBpressed) {
				PBpressed = (intr[1] >> 3) & 1;
				taglist_init (taglist);
				taglist_add (taglist, TAG_KEYCODE, KEY_POWER);
				taglist_add (taglist, TAG_KEYREPEAT, PBpressed ? 1 : 0);
				taglist_add (taglist, TAG_MODIFIER, 0);
				process_queue (KBDQUEUE, taglist);
			}
			break;
		}
#if defined(DEBUG) && PMUINTR
		printf ("\nPMU interrupt: ");
		for (i=0; i < n; i++)
			printf ("%.2x ", intr[i]);
		printf ("\n");
#endif
	}
	return TRUE; /* we assume that the filehandle for /dev/pmu can't
                    be closed from outside pbbuttonsd */
}

void
pmac_timer100 (struct tagitem *taglist)
{
	struct moddata_pmac *base = &modbase_pmac;
	struct tagitem args[6];
	int val;

	taglist_init (args);

	if (base->version == OHARE_PMU) {         /* PowerBook 3400, etc */
		if ((val = get_button(base->fd_adb, PMU_GET_VOLBUTTON)) >= 0) {
			val = val > 64 ? 100 : val * 100 / 64;
			if (base->oharevolbutton != val) {
				base->oharevolbutton = val;
				taglist_add (args, TAG_VOLUME, val);
			}
		}
		if ((val = get_button(base->fd_adb, 0x49)) >= 0) {
			val = 18 - (val >> 2);   /* This formula was derived for the usual pmac backlights */
			if (val > 15) val = 15;  /* which uses 16 brightness steps. 0=min, 15=max */
			if (val < 0) val = 0;
			if (base->oharebrightbutton != val) {
				base->oharebrightbutton = val;
				taglist_add (args, TAG_LCDBRIGHTNESS, val);
			}
		}
	}
	
	/* trackpad tapping control */
	if (base->trackpad_pause) {
		if ((--base->trackpad_pause) == 0) {
			trackpad_set_config(base->trackpad_mode);
#if defined(DEBUG) && TRACKPAD
			printf ("Trackpad tapping enabled \r");
			fflush (stdout);
#endif	
		}
	}
	
	if (args[0].tag != TAG_END)
		process_queue (CONFIGQUEUE, args);  /* distribute changes to other modules */
}

void
pmac_timer1000 (struct tagitem *taglist)
{
	struct moddata_pmac *base = &modbase_pmac;
	struct tagitem args[6];
	int val;

	taglist_init (args);

	val = base->timeleft;
	if (base->flagschanged == 0)     /* check if anybody else has already read the flags */
		base->flagschanged = pmac_update_flags (); /* if not, get ac power and cover status */
	pmac_update_batteryinfo ();

#ifndef WITH_PMUD
	/* The COVERSTATUS must be sent before POWERCHANGED because the
	 * POWERCHANGED code in module_powersave.c rely on the correct
	 * cover status.
	 */
	if (base->flagschanged & PMU_ENV_LID_CLOSED)
		taglist_add (args, TAG_COVERSTATUS, base->flags.coveropen);
#endif
	
	if (base->flagschanged & PMU_ENV_AC_POWER) {
		taglist_add (args, TAG_POWERCHANGED, base->flags.ac_power);
		if (!base->flags.ac_power)      /* running on Battery */
			base->batlog_cycle++;   /* then increase cycle */
	}
	if (val != base->timeleft && !base->flags.ac_power) {
#ifndef WITH_IBAM
		taglist_add (args, TAG_TIMECHANGED, base->timeleft);
#endif
		batlog_write(base->timeleft);
	}

	if (args[0].tag != TAG_END)
		process_queue (CONFIGQUEUE, args);  /* distribute changes to other modules */
	base->flagschanged = 0;
}


/* -------------------------------  MISCELLANEOUS FUNCTIONS ----------------------------------- */

/* This function put the machine into sleep mode. It tries pmud first. If it was not
   available, it would go directly over the PMU.
   
   Very importent is, that no modifier key is pressed, when this function is called,
   because this would confuse the input driver after wakeup. This case is not
   fully avoided, because this function is not atomic and with some fast actions
   the user could bypass our security gate.
   
   The second point is that current states of the trackpad and the keyboard will
   only stored locally. This will normally have no side effects but the global
   variables base->trackpad_mode and base->keyboard-mode won't be synchronized
   with the hardware here anymore. So the states set by external programs will
   survive sleep but also will be overwritten as soon as pbbuttonsd access one
   of the devices.
*/
void
activate_sleepmode ()
{
	struct moddata_pmac *base = &modbase_pmac;
	int tpmode, kbdmode = 0; 
#ifdef WITH_PMUD
	char buf[35];
	int fd;
#endif

	tpmode = trackpad_get_config ();
	if (base->version != OHARE_PMU)
		kbdmode = keyboard_get_config ();
	sync();
#ifndef WITH_PMUD
	/* PMU_IOC_SLEEP only succeeds if called as root.
	 * It returns immediatly if not, without putting
	 * the mashine to sleep
	 */
	ioctl (base->fd_pmu, PMU_IOC_SLEEP, 0);    /* returns after next awake */
	trackpad_set_config(tpmode);
#else
	if (((fd = tcp_open("localhost", "pmud", 879)) < 0)) {
		print_msg (PBB_WARN, _("Can't find pmud on port 879. Trigger PMU directly for sleep.\n"));
		ioctl (base->fd_pmu, PMU_IOC_SLEEP, 0);     /* returns after next awake */
		trackpad_set_config(tpmode);
	} else {
		tcp_printf(fd, "sleep\n");

	/* read some bytes, in order to prevent pmud complaining about
	 * dropped connections, before ever handling the 'sleep' command
	 */
		tcp_printf(fd, "\n");
		sleep(6);                        /* give pmud time to snooze */
		tcp_read(fd, buf, sizeof(buf));
		tcp_close(fd);
	}
#endif
	if (base->version != OHARE_PMU)
		keyboard_set_config(kbdmode);
}

/* This function gets the battery status from the PMU kernel driver via
 * /proc/pmu and fill all battery slots of moddata struct, also if less
 * batteries are present.The following flags were set accordingly:
 * ac_power, bat_present and charging.
 */
void
pmac_update_batteryinfo ()
{
	struct moddata_pmac *base = &modbase_pmac;
	FILE *fd;
	char buffer[100], *token;
	int val, n, syscurrent, time_rem = 0;
	int charge, chargemax, current, voltage, timeleft[MAX_BATTERIES];

	syscurrent = 0;
	for (n=0; n < MAX_BATTERIES; n++) {
		charge = chargemax = current = voltage = timeleft[n] = 0;
		base->flags.bat_present &= ~(1<<n);
		sprintf (buffer, "/proc/pmu/battery_%i", n);
		if ((fd = fopen (buffer, "r"))) {
			while (fgets (buffer,sizeof (buffer), fd)) {
				if ((token = strtok (buffer, ":\n"))) {
					if (!strncmp ("flags", token, 5)) {
						val = axtoi (strtok (0, ":\n"));
						base->flags.bat_present |= (val & 1) << n;
						base->flags.bat_charging = (base->flags.bat_charging & ~(1<<n)) | (((val & 2) >> 1) << n);
					} else if (!strncmp ("charge", token, 6)) {
						charge = atoi(strtok(0, ":\n"));
					} else if (!strncmp ("max_charge", token, 9)) {
						chargemax = atoi (strtok(0,":\n"));
					} else if (!strncmp ("current", token, 7)) {
						current = atoi (strtok(0, ":\n"));
					} else if (!strncmp ("voltage", token, 7)) {
						voltage = atoi (strtok(0,":\n"));
					} else if (!strncmp ("time rem.", token, 8)) {
						timeleft[n] = atoi (strtok(0, ":\n"));
					} else
						strtok (0,":\n");
				}
			}
			fclose(fd);
		}
		base->charge[n] = charge;
		base->chargemax[n] = chargemax;
		base->voltage[n] = voltage;
		syscurrent += base->current[n] = current;  /* overall current out of the batteries */
	}

 /* work around for kernel bug: After booting without batteries the kernel doesn't
    recognise the ac plug and thought by mistake the computer runs on batteries
    but there were no batteries connected. So we check for batteries here and if
	none were found it would have to be AC. */

	if (!base->flags.ac_power && !base->flags.bat_present)
		base->flags.ac_power = 1;

 /*  During plug and unplug of the AC connector it could happen that the pmu time
     remaining temporarely is set to zero. In this case we freeze the timeleft and wait
     until the pmu time remaining value returns to a valid value */

	if (syscurrent < 0) {             /* computer runs on battery */
		for (n=0; n < MAX_BATTERIES; n++) {
			if ((timeleft[n] > 0) || (base->charge[n] == 0))  /* is /proc/pmu reliable ? */
				time_rem += timeleft[n];              /* then use it */
			else if (base->version == OHARE_PMU)     /* otherwise calculate it by yourself */
				time_rem += base->charge[n] * 274 / (-syscurrent);  /* formula for old Notebooks */
			else
				time_rem += base->charge[n] * 3600 / (-syscurrent);  /* formula for modern Notebools */
		}
	} else
		time_rem = -1;

	if (time_rem != -1) {
		if (base->timeleft >= 0)
			time_rem  = base->timeleft + (time_rem - base->timeleft) / 8;   /* very simple low pass filter */
		base->timeleft = time_rem;
	}
}

/* use /proc/pmu/info as fallback, if PMU_GET_COVER did't work
 * As I know this has not happend yet and it is doubtable that
 * /proc/pmu/info will work correctly in this case. So this
 * routine may be removed in future versions.
 */	
int
pmac_get_procac ()
{
	FILE *fd;
	char buffer[100], *token;
	int ac = 0;  /* return battery if anything fails */

	if ((fd = fopen ("/proc/pmu/info","r"))) {
		while (fgets (buffer, sizeof (buffer), fd))
			if ((token = strtok (buffer,":\n"))) {
				if (!strncmp ("AC Power", token, 8))
					ac = atoi (strtok(0,":\n"));
				else
					strtok (0,":\n");
			}
		fclose(fd);
	}
	return ac;
}

/* This funtion gets some important flags from the PMU driver. If the
 * driver doesn't support the ioctl or reports an error, power source
 * will be read from /proc/pmu/info as fallback, so that power source
 * should always be set correctly.
 */
int
pmac_update_flags()
{
	struct moddata_pmac *base = &modbase_pmac;
	int env, envold, envnew;

	envold  = base->flags.ac_power ? PMU_ENV_AC_POWER : 0;
	envold |= base->flags.coveropen ? PMU_ENV_LID_CLOSED : 0;
	
	env = get_int_environment(base->fd_adb);
	if (env != -1) {
		base->flags.coveropen  = (env & PMU_ENV_LID_CLOSED) ? 0 : 1;
		base->flags.ac_power   = (env & PMU_ENV_AC_POWER) ? 1 : 0;
		base->flags.batpresent = (env & PMU_ENV_BATTERY_PRESENT) ? 1 : 0;
	} else
		base->flags.ac_power = pmac_get_procac();
	
	/* the PB G3 lombard PMU doesn't support the AC power flag in
	 * environment flags. We use the /proc entry instead.
	 */
	if (base->version < 12)
		base->flags.ac_power = pmac_get_procac();
		
	envnew  = base->flags.ac_power ? PMU_ENV_AC_POWER : 0;
	envnew |= base->flags.coveropen ? PMU_ENV_LID_CLOSED : 0;
	return (envold ^ envnew);
}

void
pmac_set_lcdbacklight (int val)
{
	struct moddata_pmac *base = &modbase_pmac;
	int ambient;
	
	if (val == -1)   /* no backlight controller available */
		return;
	
	ambient = getRawAmbient () - base->lcdfeedback;
	process_queue_single (CONFIGQUEUE, TAG_HW_BACKLIGHT, val);
	base->lcdfeedback = val ? getRawAmbient () - ambient : 0;

#if defined(DEBUG) && (TESTAMBIENT || SIMUAMBIENT)
	ambient = getRawAmbient ();
	printf("LCD Level: %2d, Raw (%4d) - LCD Feedback (%4d) = Ambient (%4d or %2d%%)\n",
		val, ambient, base->lcdfeedback, ambient - base->lcdfeedback,
		getAmbient());
	fflush(stdout);
#endif
	return;
}

/* ----------------------------  GENERIC FUNCTIONS ----------------------------- */
int
isModifier (int code, int mod)
{
	if (mod) {
		switch (code) {
			case KEY_LEFTCTRL:
			case KEY_LEFTALT:
			case KEY_LEFTSHIFT:
			case KEY_LEFTMETA:
			case KEY_RIGHTCTRL:
			case KEY_RIGHTALT:
			case KEY_RIGHTSHIFT:
			case KEY_RIGHTMETA:
			case KEY_COMPOSE:
			case KEY_CAPSLOCK:
			case KEY_NUMLOCK:
			case KEY_SCROLLLOCK:
				return 1;  /* key is modifier */
		}
	}
	return 0;  /* key is not a modifier */
}

int
haveADBKeyboard ()
{
	struct moddata_pmac *base = &modbase_pmac;
	
	if (base->machine > 0) {
		if (base->machine < 0x56)              /*  0 - 56 -> true  */
			return 1;
		if (base->machine > 0x67)              /* 68 - ?? -> false */
			return 0;
		if (((base->machine >> 4) & 0xf) == 5)
			if (base->machine >= 0x56)         /* 56 - 5f -> false */
				return 0;
		if (((base->machine >> 4) & 0xf) == 6)
			if (base->machine <= 0x67)         /* 60 - 67 -> true  */
				return 1;
	}
	return 0;
}

int
haveADBTrackpad ()
{
	struct moddata_pmac *base = &modbase_pmac;
	
	if (base->machine > 0) {
		if (base->machine < 0x56)              /*  0 - 55 -> true  */
			return 1;
		if (base->machine > 0x67)              /* 68 - ?? -> false */
			return 0;
		if (((base->machine >> 4) & 0xf) == 5)
			if (base->machine >= 0x56)         /* 56 - 5f -> false */
				return 0;
		if (((base->machine >> 4) & 0xf) == 6)
			if (base->machine <= 0x67)         /* 60 - 67 -> true  */
				return 1;
	}
	return 0;
}

int
haveI2CAmbient ()
{
	struct moddata_pmac *base = &modbase_pmac;
#if defined(DEBUG) && SIMUAMBIENT
	return 1;
#endif
	
	if ((base->machine >= 0x51) && (base->machine <= 0x57))
		if (base->lmuaddr != 0 && strlen(base->i2cdev) > 0)   
			return 1;
	return 0;
}

int
havePMUAmbient ()
{
	struct moddata_pmac *base = &modbase_pmac;

	if ((base->machine >= 0x58) && (base->machine <= 0x59))
		return 1;
	return 0;
}

/* ------------------  POWER MAC SPECIFIC KEYBOARD FUNCTIONS ------------------- */

int
keyboard_get_config ()
{
	struct moddata_pmac *base = &modbase_pmac;
	unsigned char ADBBuffer[ADB_BUFSIZE];
	int fd, n, dev, config = -1;

	if (haveADBKeyboard()) {
		if ((dev = keyboard_get_dev (base->fd_adb)) > 0) {
			if ((n = adb_read_reg (base->fd_adb, ADBBuffer, dev, 1)) > 0) {
				config = ADBBuffer[3] & 0x01 ? KEYBOARD_FNTOP : KEYBOARD_FNBACK;
			}
		}
	} else if ((fd = open (PATH_FNMODE, O_RDONLY)) >= 0) {
		if ((n = read (fd, ADBBuffer, ADB_BUFSIZE-1)) > 0) {
			ADBBuffer[n] = 0; /* terminate read buffer */
			config = atoi((const char*)ADBBuffer);
#ifdef DEBUG
			printf ("DBG: Keyboard config read: '%s' -> %d.\n", ADBBuffer, config);			
#endif
		}
		close (fd);
	}
	return config;
}

void
keyboard_set_config (int config)
{
	struct moddata_pmac *base = &modbase_pmac;
	unsigned char ADBBuffer[ADB_BUFSIZE];
	int fd, n, dev;

	if (haveADBKeyboard()) {
		if ((dev = keyboard_get_dev (base->fd_adb)) > 0) {
			n = adb_read_reg (base->fd_adb, ADBBuffer, dev, 1);
			if (n > 0) {
				ADBBuffer[3] &= 0xfe;
				ADBBuffer[3] |= (config == KEYBOARD_FNTOP) ? 0x01 : 0x00;
				adb_write_reg (base->fd_adb, ADBBuffer, n, dev, 1);
			}
		}
	} else if ((fd = open (PATH_FNMODE, O_WRONLY)) >= 0) {
		sprintf ((char*)ADBBuffer, "%d", config);
		write (fd, ADBBuffer, strlen((const char*)ADBBuffer));
#ifdef DEBUG
		printf ("DBG: Keyboard config written: '%s' -> %d.\n", ADBBuffer, config);			
#endif
		close (fd);
	}
}

int
keyboard_get_dev (int fd)
{
	unsigned char ADBBuffer[ADB_BUFSIZE];
	int n;

	for (n=1; n<16; n++) {
		if ((adb_read_reg (fd, ADBBuffer, n, 3)) > 0) {
			if (((ADBBuffer[2] & 0x0f)  == 2) &&            /* device id = keyboard */
			     ((ADBBuffer[3] & 0xc0) == 0xc0)) {        /* keyboard type */
				return n;
			}
		}
	}
	return -1;
}

/* -------------------- POWER MAC SPECIFIC TRACKPAD FUNCTIONS ------------------ */

int
trackpad_get_config ()
{
	struct moddata_pmac *base = &modbase_pmac;
	unsigned char ADBBuffer[ADB_BUFSIZE];
	int n, dev, config = -1;

	if (haveADBTrackpad()) {
		if ((dev = trackpad_get_dev (base->fd_adb)) > 0) {
			trackpad_set_prgmode (base->fd_adb, dev, 1);
			n = adb_read_reg (base->fd_adb, ADBBuffer, dev, 2);
			if (n > 0) {
				if (ADBBuffer[5] == 0xff)
					config = TRACKPAD_LOCK;
				else if (ADBBuffer[3] == 0x94)
					config = TRACKPAD_DRAG;
				else if (ADBBuffer[2] == 0x99)
					config = TRACKPAD_TAP;
				else
					config = TRACKPAD_NOTAP;
			}
			trackpad_set_prgmode (base->fd_adb, dev, 0);
		}
	}
	return config;
}

void
trackpad_set_config (int config)
{
	struct moddata_pmac *base = &modbase_pmac;
	unsigned char ADBBuffer[ADB_BUFSIZE];
	int n, dev;

	if (haveADBTrackpad()) {
		if ((dev = trackpad_get_dev (base->fd_adb)) > 0) {
			trackpad_set_prgmode (base->fd_adb, dev, 1);
			n = adb_read_reg (base->fd_adb, ADBBuffer, dev, 2);
			if (n > 0) {
				ADBBuffer[2] = (config < TRACKPAD_TAP) ? 0x19 : 0x99;
				ADBBuffer[3] = (config < TRACKPAD_DRAG) ? 0x14 : 0x94;
				ADBBuffer[5] = (config < TRACKPAD_LOCK) ? 0xb2 : 0xff;
				adb_write_reg (base->fd_adb, ADBBuffer, n, dev, 2);
			}
			trackpad_set_prgmode (base->fd_adb, dev, 0);
		}
	}
}

int
trackpad_get_dev (int fd)
{
	unsigned char ADBBuffer[ADB_BUFSIZE];
	int n;

	for (n=1; n<16; n++) {
		if ((adb_read_reg(fd, ADBBuffer, n, 1)) > 4) {
			if ((ADBBuffer[2] == 0x74) &&
			     (ADBBuffer[3] == 0x70) &&		/* 'tpad' */
			     (ADBBuffer[4] == 0x61) &&
			     (ADBBuffer[5] == 0x64) &&
				 (ADBBuffer[8] == 0x03)) {     /* class:Trackpad */
				return n;
			}
		}
	}
	return -1;
}

void
trackpad_set_prgmode (int fd, int dev, int mode)
{
	unsigned char ADBBuffer[ADB_BUFSIZE];
	int n;

	n = adb_read_reg(fd, ADBBuffer, dev, 1);
	if (n > 0) {
		ADBBuffer[8] = mode ? 0x0d : 0x03;
		adb_write_reg(fd, ADBBuffer, n, dev, 1);
		adb_read_reg(fd, ADBBuffer, dev, 2);             /* nessecary to activate program mode ! */
	}
}


/* ----------------------------- LOW LEVEL ADB FUNCTIONS ---------------------------- */

#if defined(DEBUG) && PMUTALK
/* Debug functions to track down mysterious shutdowns and freezes */
#define DEBUGLOG "/var/log/pbbuttonsd"

void
debug_wrtcmd (char *title, unsigned char *buffer, int size)
{
	struct timeval tv;
	FILE *fd;
	int i;

	gettimeofday(&tv, NULL);
	if ((fd = fopen(DEBUGLOG, "a")) != NULL) {    /* open protocol file */
		fprintf (fd, "%ld.%4d %-20s (%d)", tv.tv_sec, (int)tv.tv_usec/1000, title, size);
		for (i=0; i < size; i++) {
			fprintf (fd, " %02x", buffer[i]);
		}
		fclose(fd);
	}
}

void
debug_wrtrc (int rc, int expected)
{
	FILE *fd;

	if ((fd = fopen(DEBUGLOG, "a")) != NULL) {    /* open protocol file */
		if (rc == -1 || rc != expected)
			fprintf(fd, "\t-- FAIL\n");
		else
			fprintf(fd, "\t-- ok\n");
		fclose(fd);
	}
}

void
debug_rdcmd (char *title, unsigned char *buffer, int size)
{
	struct timeval tv;
	FILE *fd;
	int i;

	gettimeofday(&tv, NULL);
	if ((fd = fopen(DEBUGLOG, "a")) != NULL) {    /* open protocol file */
		fprintf (fd, "%ld.%4d %-20s (%d)", tv.tv_sec, (int)tv.tv_usec/1000, title, size);
		if (size >= 0) {
			for (i=0; i < size; i++) {
				fprintf (fd, " %02x", buffer[i]);
			}
			fprintf(fd, "\t-- ok\n");
		} else
			fprintf(fd, "\t-- FAIL\n");
		fclose(fd);
	}
}
/* ---------------------------------------------------------------*/
#endif

/* This function sends a request to the PMU via the ADB bus and reads
   the result. If successful the count of valid data bytes in the buffer will be
   returned, otherwise -1 */

int
send_pmu_request(int fd, unsigned char *buffer, int params, ...)
{
	va_list list;
	int n, x;

	if (params < 0 || params > 30)
		return -1;

	buffer[0] = PMU_PACKET;
	va_start(list, params);
	for (x = 0; x < params; ++x)
		buffer[x+1] = va_arg(list, int);
	va_end(list);

#if defined(DEBUG) && PMUTALK
	debug_wrtcmd("PMU request", buffer, x+1);
#endif
	n = write(fd, buffer, x+1);
#if defined(DEBUG) && PMUTALK
	debug_wrtrc(n, x+1);
#endif
	if ((n != x+1) || (n == -1))
		return -1;

	n = read(fd, buffer, ADB_BUFSIZE);
#if defined(DEBUG) && PMUTALK
	debug_rdcmd("PMU answer", buffer, n);
#endif
	if (n < 0) return -1;
	return n;
}

/* This function reads a register from an ADB device into a Buffer. The data starts
   at byte 1 in buffer. Byte 0 contains the Packettype and could be ignored.
   The funtion returns the count of valid data bytes in the buffer (excluding byte 0) */

int
adb_read_reg(int fd, unsigned char *ADBBuffer, int dev, int reg)
{
	int n;

	ADBBuffer[0] = ADB_PACKET;
	ADBBuffer[1] = ADB_READREG(dev, reg);
	
#if defined(DEBUG) && PMUTALK
	debug_wrtcmd("ADB read register", ADBBuffer, 2);
#endif
	n = write(fd, ADBBuffer, 2);
#if defined(DEBUG) && PMUTALK
	debug_wrtrc(n, 2);
#endif
	if (n != 2) return -1;
	
	n = read(fd, ADBBuffer+1, ADB_BUFSIZE-1);
#if defined(DEBUG) && PMUTALK
	debug_rdcmd("ADB read answer", ADBBuffer, n);
#endif
	return n;
}

/* This function writes a register into an ADB device. The first two bytes of the
   buffer are set by this funtion. */

int
adb_write_reg(int fd, unsigned char *ADBBuffer, int len, int dev, int reg)
{
	int n;
	len++;     /* take care about the command byte */
	ADBBuffer[0] = ADB_PACKET;
	ADBBuffer[1] = ADB_WRITEREG(dev, reg);
	
#if defined(DEBUG) && PMUTALK
	debug_wrtcmd("ADB write register", ADBBuffer, 2);
#endif
	n = write(fd, ADBBuffer, len);
#if defined(DEBUG) && PMUTALK
	debug_wrtrc(n, len);
#endif
	if (n != len) return -1;
	
	n = read(fd, ADBBuffer+1, ADB_BUFSIZE-1);
#if defined(DEBUG) && PMUTALK
	debug_rdcmd("ADB write answer", ADBBuffer, n);
#endif
	return 0;
}

/* ----------------------------- LOW LEVEL PMU FUNCTIONS ---------------------------- */

int
get_pmu_version(int fd)
{
	int count;
	unsigned char buffer[ADB_BUFSIZE];

	count = send_pmu_request(fd, buffer, 1, PMU_GET_VERSION);
	if (count == 2)
		return buffer[1];
	else
		return -1;
}

int
get_button(int fd, int button)
{
	int count;
	unsigned char buffer[ADB_BUFSIZE];

	count = send_pmu_request(fd, buffer, 1, button);
	if (count == 2)
		return buffer[1];
	else
		return -1;      /* not available or invalid */
}

int
get_int_environment(int fd)
{
	int count;
	unsigned char buffer[ADB_BUFSIZE];

	count = send_pmu_request(fd, buffer, 1, PMU_GET_COVER);
	if (count == 2)
		return buffer[1];
	else
		return -1;     /* not available or invalid */
}

/* ----------------------------- LOW LEVEL LMU FUNCTIONS ---------------------------- */

int
addPath(char *path, int maxlen, char *pattern)
{
	DIR *dh;
	struct dirent *dir;
	int rc = 1;
	
	if ((dh = opendir(path))) {
		while ((dir = readdir(dh))) {
			if ((strncmp(dir->d_name, pattern, strlen(pattern)) == 0)) {
				strncat(path, "/", maxlen-1);
				strncat(path, dir->d_name, maxlen-1);
				rc = 0;
				break;
			}			
		}
		closedir(dh);
	}
	return rc;
}

/* This function searches the device-tree for the lmu-controller to
 * read its I2C bus address. Two device paths are searched:
 *   uni-n/i2c/lmu-controller/reg or
 *   uni-n/i2c/i2c-bus/lmu-micro/reg
 *
 * If the file 'reg' is found it contents divided by two will be
 * returned. Otherwise the result is 0.
 */
int
getLMUAddress()
{
	char path[200];
	long reg;
	int fd, n, rc = 0, err = 0;

	path[0] = 0; /* terminate path buffer */
	strncat(path, OFBASE, sizeof(path)-1);
	if ((err = addPath(path, sizeof(path), "uni-n")) == 0)
		if ((err = addPath(path, sizeof(path), "i2c")) == 0)
			if ((err = addPath(path, sizeof(path), "lmu-controller")) != 0)
				if ((err = addPath(path, sizeof(path), "i2c-bus")) == 0)
					err = addPath(path, sizeof(path), "lmu-micro");
	strncat(path, "/reg", sizeof(path)-1);

	if (err == 0 && (fd = open(path, O_RDONLY)) >= 0) {
		n = read(fd, &reg, sizeof(long));
		if (n == sizeof(long))
			rc = (int) (reg >> 1);
		close(fd);
	}
	return rc;
}

int
probeLMU(char *device, int addr)
{
	char buffer[4];
	int fd, rc = 0;

	if ((fd = open(device, O_RDWR)) >= 0) {
		if (ioctl(fd, I2C_SLAVE, addr) >= 0) {
			if (read (fd, buffer, 4) == 4) 
				rc = 1;
		}
		close(fd);
	} else
		print_msg (PBB_ERR, _("Can't access i2c device %s - %s.\n"), device, strerror(errno));
	return rc;
}

/* This function tries to find the I2C device that controls the keyboard
 * illumination and the ambient light sensor used in some alluminum PowerBooks
 * It returns the device file name on success or an empty string, if no LMU
 * controller could be found.
 * The function uses the sysfs to find the appropriate i2c device to which the
 * LMU controller is connected to. The kernel module i2c-dev must be loaded
 * for this function to work.
 */
const char*
findLMUDevice(int addr)
{
	static char buffer[40];
	DIR *dh;
	struct dirent *dir;
	unsigned int n;
	int fd;

	if ((dh = opendir(SYSI2CDEV))) {
		while ((dir = readdir(dh))) {
			if (dir->d_name[0] == '.') continue;
			snprintf(buffer, sizeof(buffer), SYSI2CDEV"/%s/name", dir->d_name);
			if ((fd = open(buffer, O_RDONLY)) >= 0) {
				n = read(fd, buffer, sizeof(buffer));
				close(fd);
				if (n > 0 && n < sizeof(buffer)) {
					buffer[n-1] = 0;
					if ((strncmp(I2CCHIP" ", buffer, 6) == 0)) {
						snprintf(buffer, sizeof(buffer), "/dev/%s", dir->d_name);
						if ((probeLMU(buffer, addr)))
							break;
						buffer[0] = 0;
					}
				}
			}
		}
		closedir(dh);
	}
	return buffer;
}

/* This function returns the a compensated ambient light sensor
 * value in percent. The reduce flickering through shadows or
 * spot lights at one of the sensors the sensor with the highest
 * value is choosen.
 */
int
getAmbient ()
{
	struct moddata_pmac *base = &modbase_pmac;
	int rc, ambientmax;

	ambientmax = haveI2CAmbient() ? 1600 : 2048;
	rc = (int) (((getRawAmbient() - base->lcdfeedback) 
				* 100) / ambientmax);

	if (rc < 0)	  rc = 0;
	if (rc > 100) rc = 100;
	return rc;
}

/* The first ambient light sensor was connected to the I2C
 * Bus but recent PowerBooks 5.8 and 5.9 have a new ambient
 * light sensor connected to the PMU. This functions reads
 * the current sensor value from the right interface.
 */
int
getRawAmbient ()
{
	struct moddata_pmac *base = &modbase_pmac;
	unsigned char buf[ADB_BUFSIZE];
	int fd, rc = -1, rc2;
	
#if defined(DEBUG) && SIMUAMBIENT
	int ambient, level, idx, max;

	/* this array contains sensor values depending on the
	 * LCD brightness without additional ambient light.
	 * They simulate the LCD brightness influence on the
	 * light sensors and are for debugging only.
	 *
	 * This function simulates an I2C ambient sensor and
	 * will also work with more than 16 LCD brightness
	 * levels.
	 */
	int asens[] = {2, 182, 217, 248, 309, 337, 381, 423, 461,
		           499, 538, 576, 620, 1072, 1073, 1077};
		       
	level = process_queue_single (QUERYQUEUE, TAG_HW_BACKLIGHT, -1);
	max   = process_queue_single (QUERYQUEUE, TAG_BACKLIGHTMAX, -1);
	idx = level * sizeof(asens) / sizeof(int) / (max + 1);
	ambient = asens[level == -1 ? 0 : idx] + base->ambient;
	if (ambient > 1600) ambient = 1600;
	return ambient;
#else	
	if (haveI2CAmbient()) {
		if ((fd = open (base->i2cdev, O_RDWR)) >= 0 ) {
			if (ioctl (fd, I2C_SLAVE, base->lmuaddr) >= 0 )
				if (read (fd, buf, 4 ) == 4) {
					rc  = (buf[0] << 8) | buf[1];
					rc2 = (buf[2] << 8) | buf[3];
					rc  = rc > rc2 ? rc : rc2; 
				}
			close (fd);
		}
	}
	
	if (havePMUAmbient()) {
		if (send_pmu_request (base->fd_adb, buf, 2, 0x4F, 1) == 5) {
			rc  = (buf[2] << 8) | buf[1];
			rc2 = (buf[4] << 8) | buf[3];
			rc  = rc > rc2 ? rc : rc2; 
		}
	}
	return rc;
#endif
}

void
setKBDIllumination (unsigned short level)
{
#if !(defined(DEBUG) && SIMUAMBIENT)
	struct moddata_pmac *base = &modbase_pmac;
	unsigned char buf[ADB_BUFSIZE];
	int fd;

	if (level > KEYBLIGHTMAX) level = KEYBLIGHTMAX; /* check bounds */
	
	if (haveI2CAmbient()) {
		buf[0] = 0x01;   /* i2c register */

		/* The format appears to be:
		 *
		 *          byte 1   byte 2
		 *         |<---->| |<---->|
		 *         xxxx7654 3210xxxx
		 *             |<----->|
		 *                 ^-- brightness
		 */
		
		buf[1] = (unsigned char) level >> 4;
		buf[2] = (unsigned char) (level & 0x0F) << 4;

		if ((fd = open (base->i2cdev, O_RDWR)) >= 0 ) {
			if (ioctl (fd, I2C_SLAVE, base->lmuaddr) >= 0 )
				write (fd, buf, 3);
			close (fd);
		}
	}

	if (havePMUAmbient ()) {
		send_pmu_request (base->fd_adb, buf, 4, 0x4F, 0, 0, level);
	}
#else
	printf("KBD Level: %2d at %2d%% Ambient\n",
		level, getAmbient());
	fflush(stdout);
#endif
}

/* -------------------- LOW LEVEL OPEN FIRMWARE FUNCTIONS -------------------- */
int
getMachineID(int pmu)
{
	char buffer[32];
	int fd, n, machine = 0;

	if ((fd = open(OFBASE"/model", O_RDONLY))) {
		if ((n = read(fd, buffer, sizeof(buffer) - 1)) != -1) {
			buffer[n] = 0;   /* terminate buffer, only to be sure */
			if (strncmp("PowerBook", buffer, 9) == 0) {
				if (buffer[9] == 0)
					/* Dummy codes for pre-Lombard PowerBooks */
					machine = pmu - OHARE_PMU + 1;
				else {
					machine = (atoi(&buffer[9]) & 0xf) << 4;
					for (n = 9; buffer[n] != ',' && buffer[n] != '\0'; ++n);
					if (buffer[n] == ',')
						machine |= atoi(&buffer[n+1]) & 0xf;
				}
			}
		}
		close(fd);
	}
	return machine;
}

const char*
getMachineName(int mid)
{
	unsigned int x, y;
	static char *machines[7][10] = {
	{ N_("Unknown PowerBook"),
	  "PowerBook 3400",                        /* 0,1 */
	  "PowerBook 3500",                        /* 0,2 */
	  "PowerBook Wallstreet (Apr 1998)",       /* 0,3 */
	  NULL, NULL, NULL, NULL, NULL, NULL },
	{ NULL,
	  "PowerBook 101 Lombard (Apr 1999)",      /* 1,1 */
	  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL },
	{ NULL,
	  "iBook (Feb 2000)",                      /* 2,1 */
	  "iBook FireWire (Sep 2000)",             /* 2,2 */
	  NULL, NULL, NULL, NULL, NULL, NULL, NULL },
	{ NULL,
	  "PowerBook G3 Pismo (Feb 2000)",         /* 3,1 */
	  "PowerBook G4 Titanium (Dec 2000)",      /* 3,2 */
	  "PowerBook G4 Titanium II (Oct 2001)",   /* 3,3 */
	  "PowerBook G4 Titanium III (Apr 2002)",  /* 3,4 */
	  "PowerBook G4 Titanium IV (Nov 2002)",   /* 3,5 */
	  NULL, NULL, NULL, NULL },
	{ NULL,
	  "iBook 2 (May 2001)",                    /* 4,1 */
	  "iBook 2 (May 2002)",                   /* 4,2 */
	  "iBook 2 rev. 2 (Nov 2002)",           /* 4,3 */
	  NULL, NULL, NULL, NULL, NULL, NULL },
	{ NULL,
	  "PowerBook G4 17\" (Mar 2003)",      /* 5,1 */
	  "PowerBook G4 15\" (Sep 2003)",      /* 5,2 */
	  "PowerBook G4 17\" (Sep 2003)",      /* 5,3 */
	  "PowerBook G4 15\" (Apr 2004)",      /* 5,4 */
	  "PowerBook G4 17\" (Apr 2004)",      /* 5,5 */
	  "PowerBook G4 15\" (Feb 2005)",      /* 5,6 */
	  "PowerBook G4 17\" (Feb 2005)",      /* 5,7 */
	  "PowerBook G4 15\" (Oct 2005)",      /* 5,8 */
	  "PowerBook G4 17\" (Oct 2005)" },    /* 5,9 */
	{ NULL,
	  "PowerBook G4 12\" (Jan 2003)",      /* 6,1 */
	  "PowerBook G4 12\" (Sep 2003)",      /* 6,2 */
	  "iBook G4 (Oct 2003)",               /* 6,3 */
	  "PowerBook G4 12\" (Apr 2004)",      /* 6,4 */
	  "iBook G4 (Apr 2004)",               /* 6,5 */
	  NULL,
	  "iBook G4 (Jul 2005)",               /* 6,7 */
	  "PowerBook G4 12\" (Oct 2005)",      /* 6,8 */
	  NULL }
	};
	
	x = mid >> 4 & 0x0f;
	y = mid & 0xf;
	if (x < 7 && y < 10 && machines[x][y] != NULL)
		return machines[x][y];
	else
		return machines[0][0];
}

/* --------------------- BATTERY LOGFILE DEBUG FUNCTIONS --------------------- */
int
batlog_setup()
{
	struct moddata_pmac *base = &modbase_pmac;
	FILE *fd;
	char buffer[100], *token;
	char batlog[STDBUFFERLEN];
	int chargeold = 0, chargenew, cycle = -1;

	if (base->batlog_mode) {
		cycle = 1;
		if ((fd = fopen (DEFAULT_BATCYCLE,"r"))) {
			while (fgets (buffer, sizeof (buffer), fd))
				if ((token = strtok (buffer,":\n"))) {
					if (!strncmp ("cycle", token, 5))
						cycle = atoi (strtok(0,":\n"));
					else if (!strncmp ("charge", token, 6))
						chargeold = atoi(strtok(0, ":\n"));
					else
						strtok (0,":\n");
				}
			fclose(fd);
			if (!base->flags.ac_power) {
				chargenew = base->charge[0] + base->charge[1];
				if ((chargenew-chargeold) > 50)
					cycle++;  /* start new cycle */
			}
		}
		snprintf (batlog, STDBUFFERLEN, DEFAULT_BATLOG, cycle);
		print_msg (PBB_INFO, _("Current battery cycle: %d, active logfile: %s.\n"),
			cycle, base->batlog_mode == BATLOG_LOG ? batlog : "none");
	}
	return cycle;
}

void
batlog_save()
{
	struct moddata_pmac *base = &modbase_pmac;
	FILE *fd;

	if (base->batlog_mode)
		if ((fd = fopen (DEFAULT_BATCYCLE,"w"))) {
			fprintf (fd,"cycle   : %d\n", base->batlog_cycle);
			fprintf (fd,"charge  : %d\n", base->charge[0] + base->charge[1]);
			fclose(fd);
		}
}

void
batlog_write(long timerem)
{
	struct moddata_pmac *base = &modbase_pmac;
	struct timeval tv;
	char batlog[STDBUFFERLEN];
	FILE *fd;
	static long curtime = 0;
	int rc;

	if (base->batlog_mode != BATLOG_LOG)
		return;
		
	snprintf (batlog, STDBUFFERLEN, DEFAULT_BATLOG, base->batlog_cycle);
	rc = check_devorfile(batlog, TYPE_FILE);
	if (rc == E_NOEXIST) {
		if ((fd = fopen(batlog, "w")) != NULL) { /* open new protocol file */
			fprintf(fd, "# battery cycle %d\n# time\tvoltage[0]\tcharge[0]\tchargemax[0]\tvoltage[1]\tcharge[1]\tchargemax[1]\ttimerem.\n", base->batlog_cycle);
			fclose(fd);
			rc = 0;
		} else
			rc = E_OPEN;
	}
		
	/* to evaluate battery degradation current battery values
	   are written to a lofile for further investigation.
	   Each battery charge/discharge cycle in a seperate file. */
	if (rc == 0 && timerem > -1) {
		gettimeofday(&tv, NULL);
 		if (tv.tv_sec != curtime) {
			curtime = tv.tv_sec;
			if ((fd = fopen(batlog, "a")) != NULL) {    /* open protocol file */
				fprintf(fd, "%ld\t%d\t%d\t%d\t%d\t%d\t%d\t%ld\n", tv.tv_sec,
				  base->voltage[0], base->charge[0], base->chargemax[0],
				  base->voltage[1], base->charge[1], base->chargemax[1],
				  timerem);
   				fclose(fd);
			}
		}
	}
}
#endif /* WITH_MODULE_PMAC */
