############################################################################
##      Author Subredu Manuel <diablo@iasi.roedu.net>                      #
##                                                                         #
## This program is free software; you can redistribute it and/or modify    #
## it under the terms of the GNU General Public License as published by    #
## the Free Software Foundation; either version 2 of the License, or       #
## (at your option) any later version.                                     #
##                                                                         #
## This program is distributed in the hope that it will be useful,         #
## but WITHOUT ANY WARRANTY; without even the implied warranty of          #
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           #
## GNU General Public License for more details.                            #
##                                                                         #
## You should have received a copy of the GNU General Public License       #
## along with this program; if not, write to the Free Software             #
## Foundation, Inc., 59 Temple Place - Suite 330, Boston,                  #
## MA 02111-1307,USA.                                                      #
############################################################################

package RoPkg::Exceptions;

use strict;
use warnings;

use vars qw($VERSION);

$VERSION = '0.2.1';

use Exception::Class (
    'General' => {
      description => 'General pourpose exception',
    },
    'OutsideClass' => {
      description  => 'Method called from outside class instance',
      fields       => [ 'pkg_name' ],
    },

    'File' => {
      description => 'Files related exceptions',
      fields      => [ 'filename', 'pkg_name' ],
    },
    'File::NotFound' => {
      description => 'File cannot be found',
      isa         => 'File',
    },
    'File::Open' => {
      description => 'Could not open file',
      isa         => 'File',
    },
    'File::Create' => {
      description => 'Could not create file',
      isa         => 'File',
    },
    
    'Dir' => {
      description => 'Directories related exceptions',
      fields      => [ 'dirname', 'pkg_name' ],
    },
    'Dir::NotFound' => {
      description => 'Directory cannot be found',
      isa         => 'Dir',
    },
    'Dir::Open' => {
      description => 'Could not open directory',
      isa         => 'Dir',
    },
    'Dir::Create' => {
      description => 'Could not create directory',
      isa         => 'Dir',
    },

    'Param' => {
      description => 'Parameters related exceptions',
      fields      => [ 'name', 'pkg_name' ],
    },
    'Param::Missing' => {
      description => 'Parameter is missing or is not defined',
      isa         => 'Param',
    },
    'Param::Unknown' => {
      description => 'Unknown parameter',
      isa         => 'Param',
    },
    'Param::Wrong' => {
      description => 'Parameter has wrong value',
      isa         => 'Param',
    },

    'DB' => {
      description => 'Database related exceptions',
      fields      => [ 'dsn', 'user', 'pass', 'pkg_name' ],
    },
    'DB::Connect' => {
      description => 'Connection to database could not be established',
      isa         => 'DB',
    },
    'DB::ConnExists' => {
      description => 'Connection already exists',
      isa         => 'DB',
    },
    'DB::ConnNotFound' => {
      description => 'Connection with specified name does not exists',
      isa         => 'DB',
    },
    'DB::NoResults' => {
      description => 'No rows matched your query',
      isa         => 'DB',
    },
);

1;

__END__

=head1 NAME

RoPkg::Exceptions - exceptions used by RoPkg packages

=head1 DESCRIPTION

RoPkg::Exceptions is a collection of predefined exceptions
used by all modules from RoPkg namespace.

=head1 SYNOPSIS

 #!/usr/bin/perl
 
 use strict;
 use warnings;
 
 use RoPkg::Exceptions;
 
 sub main {
   eval {
     if ( $#ARGV == -1 ) {
       Param::Missing->throw('No parameter found');
     }
   };
 
   if ( my $e = Exception::Class->caught('Param::Missing') ) {
     print STDERR $e->message,$/,$e->trace,$/;
   }
 }
 
 main();

It is very possible that you don't find a good reason for using exceptions
from this example. And you are right. This example is kind of silly. But, try
to use exceptions in a real project. Then, you will apreciate their real value.

=head1 DEPENDENCIES

RoPkg::Exceptions require perl 5.008 or later and the Exception::Class 
module.

=head1 Exceptions List

The following exceptions are defined in this class:

=over 6

=item *) General - general pourpose exception

=item *) OutsideClass - raised when a method is called outside a class instance

=item *) File - the base for all file related exceptions . This exception has the
B<filename> parameter, who can be used to specify the path to the file. B<filename>
parameter can be used by all derivated exceptions.
The following exceptions are derivated from File:

=over 3

=item *) File::NotFound - the file was not found

=item *) File::Open - error while opening file

=item *) File::Create - could not create file

=back

=item *) Dir - the base for all directory related exceptions . This exception has the
B<dirname> parameter, who can be used to specify the path to the directory. B<dirname>
parameter can be used by all derivated exceptions.
The following exceptions are derivated from Dir:

=over 3

=item *) Dir::NotFound - the directory was not found

=item *) Dir::Open - error while opening directory

=item *) Dir::Create - could not create directory

=back

=item *) Param - the base for all parameters related exceptions. This exception has
the B<name> parameter, who can be used to specify the parameter name. B<name> can be
used by all derivated exceptions. The following exceptions are derivated from Param:

=over 3

=item *) Param::Missing - the parameter was not found

=item *) Param::Unknown - the parameter is unknown

=item *) Param::Wrong - the parameter has the wrong value or is not defined

=back

=item *) DB - the base for all database related exceptions. This exception has the
following parameters: B<dsn>, B<user> and B<pass> . This exception (and the ones
derivated from her) are mostly used in RoPkg::DB class. Of course, this doesn't
means that you can't use them in your application ;) .The following exceptions are
derivated from I<DB>:

=over 4

=item *) DB::Connect - raised when the connection with the database
could not be established

=item *) DB::ConnExists - raised when a connection with the same 
name already exists.

=item *) DB::ConnNotFound - the requested connection was not found 

=item *) DB::NoResults - the sql query returned no results

=back

=back

=head1 Exceptions Tree

=over 2

=item ROOT

=over 7

=item General

=item OutsideClass

=item File

=over 3

=item File::NotFound

=item File::Open

=item File::Create

=back

=item Dir

=over 3

=item Dir::NotFound

=item Dir::Open

=item Dir::Create

=back

=item Param

=over 3

=item Param::Missing

=item Param::Unknown

=item Param::Wrong

=back

=item DB

=over 4

=item DB::Connect

=item DB::ConnExists

=item DB::ConnNotFound

=item DB::NoResults

=back

=back

=back

=head1 SEE ALSO

L<RoPkg::Utils>

=head1 AUTHOR

Subredu Manuel <diablo@iasi.roedu.net>

=head1 VERSION

The current version is 0.2.1

=head1 DIAGNOSTICS

The tests for this module are located in B<t> directory. To
run those tests, use the following command: make test

=head1 SUBROUTINES/METHODS

This module does not have any public methods.

=head1 INCOMPATIBILITIES

None known to me

=head1 PERL CRITIC

This module is perl critic level 1 compliant with 1 exception.

=head1 BUGS AND LIMITATIONS

No known bugs. If you find one (or many) please send me 
a detailed report.

=head1 CONFIGURATION AND ENVIRONMENT

No configuration file or environment variables are used
by this module.

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2005 Subredu Manuel.  All Rights Reserved.
This module is free software; you can redistribute it 
and/or modify it under the same terms as Perl itself.
The LICENSE file contains the full text of the license.

=cut
