/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jetbrains.intellij.build

import groovy.transform.CompileStatic

/**
 * @author nik
 */
@CompileStatic
abstract class WindowsDistributionCustomizer {
  /**
   * Path to 256x256 *.ico file for Windows distribution
   */
  String icoPath

  /**
   * Path to ico file for EAP builds (if {@code null} {@link #icoPath} will be used)
   */
  String icoPathForEAP = null

  /**
   * If {@code true} *.bat files (productName.bat and inspect.bat) will be included into the distribution
   */
  boolean includeBatchLaunchers = true

  /**
   * Specify bitness of bundled JRE. If {@code null} no JRE will be bundled
   */
  JvmArchitecture bundledJreArchitecture = JvmArchitecture.x64

  /**
   * If {@code false} only 64-bit *64.exe launcher and *64.exe.vmoptions files will be created.
   */
  boolean include32BitLauncher = true

  /**
   * If {@code true} a Zip archive containing the installation with bundled Oracle JRE will be produced
   */
  boolean buildZipWithBundledOracleJre = false

  /**
   * If {@code true} a Zip archive containing the installation with bundled JetBrains RE will be produced
   */
  boolean buildZipArchive = true

  /**
   * If {@code true} Windows Installer will associate *.ipr files with the IDE in Registry
   */
  boolean associateIpr = true

  /**
   * Path to a directory containing images for installer: logo.bmp, headerlogo.bmp, install.ico, uninstall.ico
   */
  String installerImagesPath

  /**
   * List of file extensions (without leading dot) which installer will suggest to associate with the product
   */
  List<String> fileAssociations = []

  /**
   * Paths to files which will be used to overwrite the standard *.nsi files
   */
  List<String> customNsiConfigurationFiles = []

  /**
   * Path to a file which contains set of properties to manage UI options when installing the product in silent mode. If {@code null}
   * the default platform/build-scripts/resources/win/nsis/silent.config will be used.
   */
  String silentInstallationConfig = null

  /**
   * Name of the root directory in Windows .zip archive
   */
  String getRootDirectoryName(ApplicationInfoProperties applicationInfo, String buildNumber) { "" }

  /**
   * Override this method to copy additional files to Windows distribution of the product.
   * @param targetDirectory contents of this directory will be packed into zip archive and exe installer, so when the product is installed
   * it'll be placed under its root directory.
   */
  void copyAdditionalFiles(BuildContext context, String targetDirectory) {}

  /**
   * The returned name will be shown in Windows Installer and used in Registry keys
   */
  String getFullNameIncludingEdition(ApplicationInfoProperties applicationInfo) { applicationInfo.productName }

  /**
   * The returned name will be used to create links on Desktop
   */
  String getFullNameIncludingEditionAndVendor(ApplicationInfoProperties applicationInfo) { applicationInfo.shortCompanyName + " " + getFullNameIncludingEdition(applicationInfo) }

  String getUninstallFeedbackPageUrl(ApplicationInfoProperties applicationInfo) {
    return null
  }

  /**
   * The returned string is base part of a Url to tar.gz archive of JetBrains JRE x86 for windows.
   * The URL composed from two parts. The described base part and archive name (is generated by BundledJreManager.archiveNameJre).
   * The Url is used by windows installer to download and install JRE x86.
   * If the option is chosen by user JRE 64-bit and x86 will be installed together.
   */
  String getBaseDownloadUrlForJre() { return null }
}
