/*
 * (C) Copyright 2002-2003, Schlund+Partner AG
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// Local configuration
#include "config.h"

// Implementation
#include "XSLTrans.hpp"

// STDC++
#include <fstream>
#include <cassert>

// C libraries
#include <libxslt/xsltInternals.h>
#include <libxslt/transform.h>
#include <libxslt/xsltutils.h>

// Local
#include "Util.hpp"
#include "XMLTree.hpp"
#include "XMLDump.hpp"


namespace SP {
namespace GXML {

char * XSLTrans::defaultEncoding_ = "UTF-8";

// Private helper function for constructors
void
XSLTrans::genTrans(const char * xmlBuffer, int size, const std::string & baseURI)
{
	// if size < 0, we must have a valid, NULL-Terminated C-String
	if (size < 0) size = std::strlen(xmlBuffer);
	// Step 1: Create an "xmlDocPtr"
	doc_ = xmlParseMemory(xmlBuffer, size);
	if (!doc_) {
		throw(PARSE_ERR);
	}

	// Support for local includes/imports
	// See: http://mail.gnome.org/archives/xslt/2001-August/msg00037.html
	if (doc_->URL == 0 && baseURI != "")
	{
		doc_->URL = (xmlChar *)xmlMalloc(baseURI.size()+1);
		if (!doc_->URL)
		{
			xmlFreeDoc(doc_);
			throw(PARSE_ERR);
		}
		std::memcpy((void *)doc_->URL, baseURI.c_str(), baseURI.size()+1);
	}

	// Step 2: Create an "xsltStylesheetPtr"
	style_ = xsltParseStylesheetDoc(doc_);
	if (!style_) {
		xmlFreeDoc(doc_);
		throw(STYLE_ERR);
	}
}

XSLTrans::XSLTrans(const char * xmlBuffer, int size, const std::string & baseURI)
{
	genTrans(xmlBuffer, size, baseURI);
}

XSLTrans::XSLTrans(const std::string & xmlString, const std::string & baseURI)
{
	genTrans(xmlString.c_str(), -1, baseURI);
}

XSLTrans::XSLTrans(std::ifstream & f, const std::string & baseURI)
{
	genTrans(istream2String(f).c_str(), -1, baseURI);
}

XSLTrans::~XSLTrans()
{
	// This frees doc_ as well
	xsltFreeStylesheet(style_);
}

// Trans into XMLTree
std::auto_ptr<XMLTree>
XSLTrans::transToXMLTree(const xmlDocPtr doc) const
{
	assert(doc);
	xmlDocPtr resultTree = xsltApplyStylesheet(style_, doc, 0);
	if (!resultTree) {
		throw(TRANS_ERR);
	}
	return std::auto_ptr<XMLTree>(new XMLTree(resultTree));
}

// This is the master trans method
std::auto_ptr<XMLDump>
XSLTrans::trans(const xmlDocPtr doc) const
{
	assert(doc);
	xmlDocPtr resultTree = xsltApplyStylesheet(style_, doc, 0);
	if (!resultTree) {
		throw(TRANS_ERR);
	}
	XSLTOutputBuf * obuf = new XSLTOutputBuf(resultTree, style_);
	xmlFreeDoc(resultTree);

	XMLDump * result = new XMLDump(obuf);
	return std::auto_ptr<XMLDump>(result);
}

std::auto_ptr<XMLDump>
XSLTrans::trans(const XMLTree * xmlTree) const
{
	assert(xmlTree);
	return trans(xmlTree->getDocPtr());
}

std::auto_ptr<XMLDump>
XSLTrans::trans(const std::string & xmlString) const
{
	XMLTree xmlTree(xmlString);
	return trans(&xmlTree);
}

xsltStylesheetPtr
XSLTrans::getStylesheetPtr()
{
	return style_;
}

}}
